# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
from collections import defaultdict
from datetime import date, timedelta, datetime
from itertools import islice
from optparse import OptionParser, make_option

import pytz
import requests
from django.conf import settings
from django.db import transaction, IntegrityError

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.schedule import RouteSchedule
from travel.avia.library.python.common.models.transport import TransportModel
from travel.avia.library.python.common.utils.date import RunMask
from travel.avia.library.python.common.utils.iterrecipes import pairwise

from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler

log = logging.getLogger(__name__)


def _main():
    log.info('Downloading the route schedule')

    rthreads = []

    url = 'https://{}/rest/dyn/rthreads/?page_size={}'.format(
        settings.REST_HOST, 2000
    )
    while url:
        data = requests.get(url, verify=False).json()
        rthreads.extend(data['results'])
        url = data['next']

    log.info('Received %d rthreads', len(rthreads))

    today = date.today()
    run_masks = defaultdict(lambda: RunMask(today=today, strict=True))

    for rthread in rthreads:
        rthread_run_time = datetime.strptime(
            rthread['tz_start_time'], '%H:%M:%S'
        ).time()

        rthread_run_datetimes = [
            datetime.combine(day, rthread_run_time)
            for day in RunMask(
                rthread['year_days'], today=today, strict=True
            ).iter_dates()
        ]

        for rtstation_from, rtstation_to in pairwise(rthread['rtstation_set']):
            station_run_datetimes = (
                dt + timedelta(minutes=rtstation_from['tz_departure'])
                for dt in rthread_run_datetimes
            )

            rtstation_tz = rtstation_from['time_zone']
            station_tz = rtstation_from['station']['time_zone']

            if rtstation_tz and station_tz and rtstation_tz != station_tz:
                try:
                    station_run_datetimes = [
                        pytz.timezone(rtstation_tz)
                            .localize(run_time)
                            .astimezone(pytz.timezone(station_tz))
                            .replace(tzinfo=None)
                        for run_time in station_run_datetimes
                    ]
                except Exception:
                    log.exception('Bad timezone on rthread %r', rthread)
                    continue

            run_masks[
                rtstation_from['station']['id'],
                rtstation_to['station']['id'],
                rthread['number'].strip(),
                rthread['t_model_id'],
            ] |= RunMask(today=today, days=station_run_datetimes, strict=True)

    log.info('Prepared %d schedule entries', len(run_masks))

    station_id_set = set(Station.objects.values_list('id', flat=True))
    t_model_id_set = set(TransportModel.objects.values_list('id', flat=True))

    models = (
        RouteSchedule(
            station_from_id=from_id,
            station_to_id=to_id,
            route_number=route,
            t_model_id=(
                model_id
                if model_id in t_model_id_set
                else None
            ),
            run_mask=str(run_mask),
        )
        for (from_id, to_id, route, model_id), run_mask in run_masks.iteritems()
        if {from_id, to_id}.issubset(station_id_set)
    )

    with transaction.atomic():
        RouteSchedule.objects.all().delete()
        while True:
            batch = list(islice(models, 1000))
            if not batch:
                break
            try:
                RouteSchedule.objects.bulk_create(batch)
            except IntegrityError as exc:
                log.exception('IntegrityError on batch: %r', exc)
                for model in batch:
                    try:
                        model.save()
                    except IntegrityError as model_exc:
                        log.exception('IntegrityError on model: %r', model_exc)

    log.info('Saved the new route schedule')


def main():
    options, _ = OptionParser(
        option_list=[make_option('-v', '--verbose', action='store_true')]
    ).parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    try:
        _main()
    except Exception as exc:
        log.exception('Route schedule sync exception: %r', exc)
        raise
