# -*- encoding: utf-8 -*-
import logging
import os
from datetime import datetime, timedelta
from optparse import OptionParser

from library.python import resource
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder

ALLOWED_ENVS = ['production']

PARTNER_CODE = 'aeroflot'
PARTNER_BILLING_ID = 1026
CONVERSION = 179.17

AEROFLOT_ECPC_YQL_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'aeroflot_ecpc.sql')

logger = logging.getLogger(__name__)


def do_query(yql_client, query_date):
    query = resource.find(AEROFLOT_ECPC_YQL_FILE)

    yql_query = yql_client.query(query,  syntax_version=1)
    yql_query.run(
        parameters=YqlParameterValueBuilder.build_json_map({
            '$query_date': YqlParameterValueBuilder.make_string(query_date.strftime('%Y-%m-%d')),
            '$partner_code': YqlParameterValueBuilder.make_string(PARTNER_CODE),
            '$billing_order_id': YqlParameterValueBuilder.make_int32(PARTNER_BILLING_ID),
            '$conversion': YqlParameterValueBuilder.make_float(CONVERSION),
        }),
    )
    return yql_query


def update_cpc_price(result):
    from travel.avia.library.python.common.models.partner import CPCPrice

    logger.info('Update eCPC price')

    attr_map = {
        'direct_rasp': 'eCPC_rasp_direct',
        'direct_sovetnik': 'eCPC_sovetnik_direct',
        'direct_wizard': 'eCPC_wizard_direct',
        'indirect': 'eCPC',
        'indirect_wizard': 'eCPC_wizard_indirect',
    }

    cpc_price = CPCPrice.objects.get(partner__code=PARTNER_CODE, national_version__code='ru')
    for result_key, cpc_price_attr in attr_map.items():
        value = result.get(result_key)
        if value is not None:
            setattr(cpc_price, cpc_price_attr, value)
    cpc_price.save()


def main():
    import travel.avia.admin.init_project  # noqa

    from django.conf import settings

    from travel.avia.admin.avia_scripts.utils.environment import enabled_env
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yql_helpers import get_yql_operation_url, log_errors

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(logger)
    create_current_file_run_log()

    if enabled_env(settings.ENVIRONMENT, ALLOWED_ENVS, logger):
        logger.info('Start')

        try:
            yql_client = YqlClient(token=settings.YQL_TOKEN)

            yesterday = datetime.today() - timedelta(days=1)
            yql_query = do_query(yql_client, yesterday)
            logger.info('YQL operation: %s', get_yql_operation_url(yql_query))

            yql_query.wait_progress()

            if yql_query.is_ok:
                logger.info('YQL query is done')
                result = {}
                for table in yql_query.get_results():
                    table.fetch_full_data()
                    for row in table.rows:
                        result[row[1]] = row[4]
                update_cpc_price(result)
            else:
                log_errors(yql_query, logger)
        except Exception as e:
            logger.exception('Job "Update CPC Price" failed with "{}"'.format(str(e)))

        logger.info('End')
