import travel.avia.admin.init_project  # noqa

import argparse
import json
import logging
import os
import sys
from collections import defaultdict
from copy import copy
from datetime import datetime, timedelta
from urlparse import urljoin

import requests
from django.conf import settings

import yt.wrapper as yt
from library.python import resource
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

from travel.avia.admin.lib import yql_helpers
from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler
from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric


logger = logging.getLogger(__name__)


TARGET_DAY = 40
ONE_DAY_EXP_DEFAULTS = {
    'min_forward_delta': TARGET_DAY,
    'max_forward_delta': TARGET_DAY,
}
EXPERIMENTS = {
    'one_day': dict(ONE_DAY_EXP_DEFAULTS, national_version='ru', experiment='one_day'),
    'one_day_kz': dict(ONE_DAY_EXP_DEFAULTS, national_version='kz', experiment='one_day_kz'),
    'one_day_tr': dict(ONE_DAY_EXP_DEFAULTS, national_version='tr', lang='tr', experiment='one_day_tr'),
}
QUERY_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'get_popular_directions.sql')
ALLOWED_ENVS = ['production', 'dev']


class YeahError(Exception):
    pass


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-d', '--redirect-days', dest='redirect_days', default=30, type=int, help='Number of days to take redirects')
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('-n', default=24000, type=int)
    parser.add_argument('--dry-run', action='store_true', help='do not update config')
    parser.add_argument('--redirect-threshold', default=1, type=int, help='minimum number of redirects for direction to get in heater list')

    return parser.parse_args()


def get_config_directory(environment):
    if environment == 'production':
        return '//home/avia/yeah_configs'

    return '//home/avia/{}/yeah_configs'.format(environment)


def read_most_popular_by_nv(ytc, table):
    popular_by_nv = defaultdict(list)
    for r in ytc.read_table(table, format=yt.JsonFormat()):
        popular_by_nv[r['national_version']].append((r['from_id'], r['to_id']))
    return dict(popular_by_nv)


def generate_document(directions):
    return {
        'docs': [{
            'DirectionsList': {
                'type': '#p',
                'value': json.dumps(directions),
            },
            "merge_key": [{
                "type": "#k",
                "value": "TopDirections"
            }],
            "url": "TopDirections_force"
        }],
        'action': 'modify',
        'prefix': 0,
    }


def get_yeah_path(config_name):
    return urljoin(settings.YEAH_URL, 'heater_config/custom/{}'.format(config_name))


def post_config(directions, config_name):
    if not settings.YEAH_URL:
        logger.warning('YEAH_Url is not set')
    else:
        response = requests.post(
            urljoin(settings.YEAH_URL, 'heater_config/custom/{}'.format(config_name)),
            json=directions
        )
        if response.status_code != 200:
            logger.error(
                'YEAH Error. Status code: %d, message: %r',
                response.status_code, response.content
            )


def save_config_to_yt(yt_client, config, experiment_name):
    destination_table = yt.ypath_join(
        get_config_directory(settings.ENVIRONMENT),
        experiment_name,
        datetime.today().strftime('%Y-%m-%d')
    )
    logger.info('Save config to %s', destination_table)
    with yt_client.Transaction():
        if yt_client.exists(destination_table):
            yt_client.remove(destination_table)

        yt_client.create(
            'table',
            destination_table,
            recursive=True,
            attributes={
                'optimize_for': 'scan',
                'scema': [
                    {'name': 'travel_time', 'type': 'int64'},
                    {'name': 'experiment', 'type': 'string'},
                    {'name': 'req_count', 'type': 'int64'},
                    {'name': 'code_to', 'type': 'string'},
                    {'name': 'code_from', 'type': 'string'},
                    {'name': 'min_forward_delta', 'type': 'int64'},
                    {'name': 'max_forward_delta', 'type': 'int64'},
                ]
            })

        yt_client.write_table(destination_table, config)


def get_current_config(exp_name):
    if not settings.YEAH_URL:
        logger.warning('YEAH_Url is not set')
        return []

    response = requests.get(get_yeah_path(exp_name))
    if response.status_code != 200:
        raise YeahError('Can not get yeah config. Status code: {}. Message: {}.'.format(
            response.status_code,
            response.content,
        ))

    return response.json().get('data', [])


def merge_configs(old, new, max_directions):
    logger.info('Old directions: %d', len(old))
    logger.info('New directions: %s', len(new))

    if len(old) >= max_directions:
        logger.info('Old directions take all quota')
        return old

    used_directions = set(
        (r['code_from'], r['code_to'])
        for r in old
    )

    merged = copy(old)
    for new_direction in new:
        key = new_direction['code_from'], new_direction['code_to']
        if key not in used_directions:
            merged.append(new_direction)
            used_directions.add(key)

            if len(merged) >= max_directions:
                break

    logger.info('In merged: %d', len(merged))
    return merged


def update_heater_conf(directions_by_nv, yt_client, max_directions):
    for exp_name, exp_config in EXPERIMENTS.iteritems():
        national_version = exp_config['national_version']
        current_config = get_current_config(exp_name)

        new_directions = [
            dict(
                exp_config,
                travel_time=0,
                req_count=1,
                code_from=code_from,
                code_to=code_to,
            ) for code_from, code_to in directions_by_nv.get(national_version, [])
        ]
        new_config = merge_configs(current_config, new_directions, max_directions)
        save_config_to_yt(yt_client, new_config, exp_name)
        post_config(new_config, exp_name)


def main():
    args = parse_args()

    if args.verbose:
        add_stdout_handler(logger)

    create_current_file_run_log()

    logger.info('Start')
    if settings.ENVIRONMENT not in ALLOWED_ENVS:
        logger.info('Can work only in %s', ', '.join(ALLOWED_ENVS))
        return

    yql_client = YqlClient(token=settings.YQL_TOKEN)
    ytc = AviaYtClientFabric().create()
    right_date = (datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')
    left_date = (datetime.now() - timedelta(days=1 + args.redirect_days)).strftime('%Y-%m-%d')

    temp_table = ytc.create_temp_table()
    query = resource.find(QUERY_FILE)
    operation = yql_client.query(
        query,
        title='[YQL] Heater config updating',
        syntax_version=1
    )

    operation.run(
        parameters=ValueBuilder.build_json_map({
        '$StartDate': ValueBuilder.make_string(right_date),
        '$EndDate': ValueBuilder.make_string(right_date),
        '$StartRedirectDate': ValueBuilder.make_string(left_date),
        '$EndRedirectDate': ValueBuilder.make_string(right_date),
        '$NDirections': ValueBuilder.make_int64(args.n),
        '$OutputTable': ValueBuilder.make_string(temp_table),
        '$MinNumberOfRedirects': ValueBuilder.make_int64(args.redirect_threshold),
    }))
    logger.info('YQL operation: %s', operation.share_url)

    operation.wait_progress()

    if not operation.is_success:
        yql_helpers.log_errors(operation, logger)
        sys.exit(1)
    most_popular_by_nv = read_most_popular_by_nv(ytc, temp_table)

    if args.dry_run:
        logger.info('Skip config updating')
    else:
        update_heater_conf(most_popular_by_nv, ytc, args.n)
    logger.info('End')
