"""
Script for updating svg images in MDS
See https://st.yandex-team.ru/RASPTICKETS-15971 for details
"""

import travel.avia.admin.init_project  # noqa

import argparse
import cStringIO
import logging

import requests
from django.core.files import File
from django.db import connection

from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant
from travel.avia.library.python.avia_data.storage import AvatarsClient, AVATAR_READ_URL, AVATAR_WRITE_URL
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.models.schedule import Company

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log


logger = logging.getLogger(__name__)


class ImageUpdater(object):
    def __init__(self, avatars_client, native_mode, logger):
        self._avatars_client = avatars_client
        self._logger = logger
        self._native_mode = native_mode

        self._summary = []

    def update_instance_field(self, instance, field):
        self._logger.info('Update %r field %s', instance, field)
        image_object = getattr(instance, field)
        if not image_object:
            self._logger.info('Image object is empty, skip')
            return {}

        url = image_object.url
        if not url:
            self._logger.info('Url is empty, skip')
            return {}

        response = requests.get(url)
        response.raise_for_status()
        if self._native_mode:
            stream = cStringIO.StringIO(response.content)
            stream.seek(0)

            image_object.save(image_object.name, File(stream))
            return {field: image_object.url}

        new_url = self._avatars_client.put(image_object.name, response.content)
        self._logger.info('Success')
        return {field: new_url}

    def update_instance_fields(self, instance, fields):
        instance_update = {}
        for field in fields:
            try:
                instance_update.update(self.update_instance_field(instance, field))
            except Exception:
                self._logger.exception('ERROR:')
                self._summary.append((instance.id, field))

        if instance_update:
            return {instance.id: instance_update}

        return {}

    def update_model_fields(self, model, fields):
        self._summary = []

        model_update = {}
        self._logger.info('Update model %r fields %s', model, fields)
        for instance in model.objects.all():
            model_update.update(self.update_instance_fields(instance, fields))

        if model_update and not self._native_mode:
            self._save_model(model, model_update)

        self._logger.info('Finish update model %r', model)
        return self._summary

    def _save_model(self, model, model_update):
        if not model_update:
            return

        table_name = model._meta.db_table
        queries = []
        for instance_id, instance_update in model_update.iteritems():
            if not instance_update:
                continue

            prefix = 'UPDATE {table_name} SET '.format(
                table_name=table_name,
            )

            query = prefix + ', '.join(
                '{field}=\'{value}\''.format(
                    field=field,
                    value=value,
                )
                for field, value in instance_update.iteritems()
            ) + ' WHERE id = {instance_id};'.format(instance_id=instance_id)

            queries.append(query)

        single_update_query = '\n'.join(queries)
        self._logger.info('Ready to execute %s', single_update_query)
        connection.cursor().execute(single_update_query, [])


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('--native-mode', action='store_true', help='If enabled, use native Django field saving routines instead of directly changing database')

    return parser.parse_args()


def main():
    create_current_file_run_log()

    args = parse_args()

    if args.verbose:
        add_stdout_handler(logger)

    partner_logos = ['logo2_svg_{}'.format(nv) for nv in ['ru', 'ua', 'kz', 'tr', 'com']]
    to_update = [
        (Partner, partner_logos),
        (AmadeusMerchant, partner_logos),
        (DohopVendor, partner_logos),
        (Company, ['svg_logo2']),
    ]

    updater = ImageUpdater(
        AvatarsClient(namespace='put-avia', read_url=AVATAR_READ_URL, write_url=AVATAR_WRITE_URL, logger=logger),
        native_mode=args.native_mode,
        logger=logger,
    )

    summary = {}
    for model, fields in to_update:
        summary[model] = updater.update_model_fields(model, fields)

    for model, model_summary in summary.iteritems():
        logger.info('During updating model %r, model was not updated', model)
        for instance_id, field in model_summary:
            logger.info('Id=%d, field=%s', instance_id, field)
