# -*- encoding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
from datetime import datetime
from itertools import groupby
from subprocess import check_call

from django.conf import settings

from travel.avia.library.python.common.models.partner import Partner
from travel.avia.admin.stats.models import ImportChart
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)


def rsync_from_space(log_name):
    share = '%s/tickets/' % settings.SPACE_RASPADMIN_SHARE

    log.info('Rsync %s from %s', log_name, share)

    try:
        check_call('mkdir -p /tmp/space_log > /dev/null 2>&1', shell=True)

        check_call(
            'rsync -vrtDzu %s%s /tmp/space_log/ > /dev/null 2>&1' % (
                share, log_name
            ),
            shell=True
        )

    except Exception:
        log.exception('rsync')

        return False

    return True


def update_from(log_name, partner_getter):
    log.info('Update importcharts from %s', log_name)

    created_dates = []

    try:
        max_when = ImportChart.objects.order_by('-when')[0].when

    except IndexError:
        max_when = datetime.min

    skip_traceback_write = False

    for row in open(log_name):
        try:
            when, event, partner_code, value = row.split()

            try:
                when = datetime.strptime(when, '%Y-%m-%d_%H')

            except ValueError:
                # Запишем только первый трейсбек
                if not skip_traceback_write:
                    log.exception('%s', row)
                    skip_traceback_write = True

                continue

            # Обновлять будем только записи последнего часа
            if when < max_when:
                continue

            partner = partner_getter(partner_code)

            if not partner:
                continue

            chart, created = ImportChart.objects.get_or_create(
                partner=partner,
                when=when,
                event=event,
                machine='',
            )

            if created:
                created_dates.append(when.date())

            chart.value = float(value)

            chart.save()

        except Exception:
            log.exception('%s', row)

    for when_dt, items in groupby(sorted(created_dates)):
        log.info(
            'Updated from %s: created %s objects for %s',
            log_name, len(list(items)), when_dt
        )


class PartnerGetter(object):
    def __init__(self):
        self._cache = {}

    def __call__(self, code):
        if code not in self._cache:
            try:
                self._cache[code] = self.get(code)

            except Exception:
                self._cache[code] = None
                log.warning('Partner not found [%s]', code)

        return self._cache[code]


class FlowlogPartnerGetter(PartnerGetter):
    def get(self, code):
        return Partner.objects.get(code=code)


class ShowlogPartnerGetter(PartnerGetter):
    def get(self, code):
        return Partner.objects.get(billing_datasource_id_production=code)


def _main():
    logs = {
        'flow.log': FlowlogPartnerGetter(),
        'show.log': ShowlogPartnerGetter(),
    }

    for log_name, partner_getter in logs.items():
        if rsync_from_space(log_name):
            update_from('/tmp/space_log/%s' % log_name, partner_getter)


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    _main()

    log.info('Done')
