# -*- encoding: utf-8 -*-
import calendar
import functools

from datetime import datetime, date
from optparse import Option, OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


def yt_last_logs_tables(path, days):
    tables = []
    for table in yt.search(path, node_type="table"):
        tables.append((
            datetime.strptime(table.split('/')[-1], '%Y-%m-%d'),
            table
        ))
    sorded_tables = sorted(tables, reverse=True)[:days]

    return [t[1] for t in sorded_tables]


def get_hour_range(min_date, max_date):
    return min_date, max_date


def year_range(min_date, max_date):
    return (
        date(min_date.year, 1, 1),
        date(max_date.year, 12, 31)
    )


def month_range(min_date, max_date):
    month_days = calendar.monthrange(max_date.year, max_date.month)

    return (
        date(min_date.year, max_date.month, 1),
        date(max_date.year, max_date.month, month_days[1])
    )


def day_range(min_date, max_date):
    return (
        date(min_date.year, min_date.month, min_date.day),
        date(max_date.year, max_date.month, max_date.day)
    )


def hour_range(min_date, max_date):
    return min_date, max_date


def period_range(min_date, max_date):
    kwargs = {
        'min_date': min_date,
        'max_date': max_date,
    }

    period_range = {
        'hour': hour_range(**kwargs),
        'day': day_range(**kwargs),
        'month': month_range(**kwargs),
        'year': year_range(**kwargs),
    }

    return period_range


def main():
    import travel.avia.admin.init_project  # noqa

    import logging
    from django.conf import settings

    from django.db.models import Count, Sum
    from django.db import transaction  # noqa

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper

    from travel.avia.admin.stats.models import ServiceMetric
    from travel.avia.library.python.common.models.partner import Partner

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def aggregate_periods(created_dates):
        PERIODS_SELECTORS = [
            ["day",  {"when_date": "DATE(`when`)"}],
            ["month", {"when_date": "DATE(DATE_FORMAT(`when`, '%%Y-%%m-01'))"}],
            ["year", {"when_date": "DATE(DATE_FORMAT(`when`, '%%Y-01-01'))"}]
        ]

        SERVICEMETRICS_PERIODS = ['hour', 'day', 'month', 'year']

        if created_dates:
            min_date = min(created_dates)
            max_date = max(created_dates)

            for period, selector in PERIODS_SELECTORS:
                period_min_date, period_max_date = period_range(min_date, max_date)[period]
                max_datetime = datetime(period_max_date.year, period_max_date.month, period_max_date.day, 23, 59, 59)
                previous_period = SERVICEMETRICS_PERIODS[SERVICEMETRICS_PERIODS.index(period)-1]

                log.info('Aggregate all service metrics for a "%s" in %s\'s range from %s to %s' % (
                    period, previous_period, period_min_date, max_datetime)
                )

                data_source = ServiceMetric.objects.filter(
                    when__range=[period_min_date, max_datetime],
                    period=previous_period,
                ).extra(
                    select=selector
                ).values(
                    'when_date', 'service', 'group', 'event'
                ).annotate(
                    Count('service'), Count('group'), Count('event'), sum_values=Sum('value')
                )

                for row in data_source:
                    metric, created = ServiceMetric.objects.get_or_create(
                        when=row['when_date'],
                        period=period,
                        service=row['service'],
                        group=row['group'],
                        event=row['event']
                    )

                    metric.value = float(row['sum_values'])
                    metric.save()

    def precache_partners():
        avia_partners = Partner.objects.filter(t_type__code='plane')

        return [p.code for p in avia_partners]

    def extract_partner(record):
        return record.get('partner') or record['importer'].split('[')[0]

    def cache_map(avia_partners, record):
        service = record.get('service')
        if extract_partner(record) in avia_partners and service in ['ticket', 'rasp']:
            iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
            key = iso_eventtime.strftime("%Y-%m-%d_%H")

            yield {
                'key': key,
                'service': service,
                'count': 1
            }

    def cache_reduce(key, records):
        yield {
            'key': key['key'],
            'service': key['service'],
            'count': sum([int(r['count']) for r in records])
        }

    def cache_map_reduce(search_tables, tmp_table, avia_partners):
        yt.run_map_reduce(
            mapper=functools.partial(cache_map, avia_partners),
            reducer=cache_reduce,
            source_table=search_tables,
            destination_table=tmp_table,
            sort_by=['key', 'service'],
            reduce_by=['key', 'service'],
            format=yt.DsvFormat()
        )

    def write_cache_results(tmp_search_table):
        log.info('Write results from %s' % tmp_search_table)

        for fields in yt.read_table(tmp_search_table, format=yt.DsvFormat(), raw=False):
            count = float(fields['count'])
            when = datetime.strptime(fields['key'], '%Y-%m-%d_%H')
            service = fields['service']

            metric, created = ServiceMetric.objects.get_or_create(
                when=when,
                period='hour',
                service=service,
                group='daemon',
                event='cache_hit'
            )

            created_dates.add(when.date())

            metric.value = count
            metric.save()

    def write_miss_results(tmp_search_table):
        log.info('Write results from %s' % tmp_search_table)

        for fields in yt.read_table(tmp_search_table, format=yt.DsvFormat(), raw=False):
            count = float(fields['count'])
            when = datetime.strptime(fields['key'], '%Y-%m-%d_%H')
            service = fields['serivice']

            metric, created = ServiceMetric.objects.get_or_create(
                when=when,
                period='hour',
                service=service,
                group='daemon',
                event='cache_miss'
            )

            created_dates.add(when.date())

            metric.value = count
            metric.save()

    # IP / YANDEXUID
    def ip_map(record):
        service = record.get('service')

        if 'userip' in record and 'yandexuid' in record and service in ['ticket', 'rasp']:
            iso_eventtime = datetime.strptime(record['iso_eventtime'], '%Y-%m-%d %H:%M:%S')
            key = iso_eventtime.strftime("%Y-%m-%d_%H")

            yield {
                'key': key,
                'ip': record['userip'],
                'yandexuid': record['yandexuid'],
                'service': service,
            }

    def ip_reduce(key, records):
        ips = set()
        yandexuids = set()
        count = 0

        for r in records:
            ips.add(r['ip'])
            yandexuids.add(r['yandexuid'])
            count += 1

        yield {
            'key': key['key'],
            'service': key['service'],
            'count': count,
            'ip': len(ips),
            'yandexuid': len(yandexuids),
        }

    def ip_map_reduce(search_tables, tmp_table):
        yt.run_map_reduce(
            mapper=ip_map,
            reducer=ip_reduce,
            source_table=search_tables,
            destination_table=tmp_table,
            sort_by=['key', 'service'],
            reduce_by=['key', 'service'],
            format=yt.DsvFormat()
        )

    def write_ip_results(tmp_search_table):
        log.info('Write results from %s' % tmp_search_table)

        for fields in yt.read_table(tmp_search_table, format=yt.DsvFormat(), raw=False):
            ip_count = float(fields['ip'])
            yandexuid_count = float(fields['yandexuid'])
            count = float(fields['count'])
            when = datetime.strptime(fields['key'], '%Y-%m-%d_%H')
            service = fields['service']

            ip_metric, _created = ServiceMetric.objects.get_or_create(
                when=when,
                period='hour',
                service=service,
                group='form',
                event='ip'
            )

            ip_metric.value = ip_count
            ip_metric.save()

            uid_metric, _created = ServiceMetric.objects.get_or_create(
                when=when,
                period='hour',
                service=service,
                group='form',
                event='yandexuid_request'
            )

            uid_metric.value = yandexuid_count
            uid_metric.save()

            metric, created = ServiceMetric.objects.get_or_create(
                when=when,
                period='hour',
                service=service,
                group='form',
                event='user_search'
            )

            metric.value = count
            metric.save()

            created_dates.add(when.date())

    created_dates = set()

    configure_wrapper(yt)

    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    tmp_search_table = yt.create_temp_table(
        path=settings.TOP_DIRECTIONS_TMP_PATH,
        prefix='rasp_users_search_'
    )

    tmp_cache_table = yt.create_temp_table(
        path=settings.TOP_DIRECTIONS_TMP_PATH,
        prefix='rasp_users_search_'
    )

    tmp_miss_table = yt.create_temp_table(
        path=settings.TOP_DIRECTIONS_TMP_PATH,
        prefix='rasp_users_search_'
    )

    avia_partners = precache_partners()
    log.info('%s partners was cached' % len(avia_partners))

    # RASP-USER-SEARCH-LOG
    search_tables = yt_last_logs_tables('//home/rasp/logs/rasp-users-search-log', 2)
    ip_map_reduce(search_tables, tmp_search_table)
    write_ip_results(tmp_search_table)
    yt.remove(tmp_search_table)

    # RASP-PARTNERS-CACHE-LOG
    cache_tables = yt_last_logs_tables('//home/rasp/logs/rasp-partners-cache-log', 2)
    cache_map_reduce(cache_tables, tmp_cache_table, avia_partners)
    write_cache_results(tmp_cache_table)
    yt.remove(tmp_cache_table)

    # RASP-PARTNERS-QUERY-LOG
    miss_tables = yt_last_logs_tables('//home/rasp/logs/rasp-partners-query-log', 2)
    cache_map_reduce(miss_tables, tmp_miss_table, avia_partners)
    write_miss_results(tmp_miss_table)
    yt.remove(tmp_miss_table)

    aggregate_periods(created_dates)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES
