# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
import os
import random
import uuid
from datetime import datetime, timedelta
from functools import partial
from operator import itemgetter
from optparse import OptionParser
import multiprocessing.dummy as multiprocessing
from urllib import urlencode

import requests
from django.conf import settings
from django.template import loader
from retrying import retry

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.avia_data.models import SettlementBigImage
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.utils.iterrecipes import pairwise, group_by
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.admin.management.commands.run_task import script_runner


log = logging.getLogger(__name__)
ALLOWED_ENVS = ['production', 'dev']
months = (
    u'январе', u'феврале', u'марте', u'апреле', u'мае', u'июне',
    u'июле', u'августе', u'сентябре', u'октябре', u'ноябре', u'декабре',
)
MONTHS_MAPPING = dict(enumerate(months, start=1))

UTM = {
    'utm_source': 'doubletrade',
    'utm_medium': 'feed',
    'utm_campaign': 'ru',
}

TOKEN = settings.TRADEDOUBLER_TOKEN
FEED_ID = 25365


class ImageCache(object):
    def __init__(self):
        qs = SettlementBigImage.objects.exclude(url2='')

        def url(base_url):
            return '{}/orig'.format(base_url)

        self._dummy_urls = tuple(
            url(u[0]) for u in qs.filter(
                settlement__isnull=True
            ).values_list('url2')
        )
        self._urls = {
            _id: url(base_url)
            for _id, base_url in qs.filter(
                settlement__isnull=False
            ).values_list('settlement_id', 'url2')
        }

    def url(self, settlement_id):
        return self._urls.get(settlement_id, random.choice(self._dummy_urls))


IMAGES = ImageCache()


def xgettext_month_prepositional(month, lang):
    # todo: вынести сюда from travel.avia.admin.xgettext.common import xgettext_month
    return MONTHS_MAPPING[int(month)]


def product_params(min_price):
    point_from = min_price['departure_settlement']
    point_to = min_price['arrival_settlement']
    date_fwd = min_price['date_forward']
    date_bwd = min_price['date_backward']

    def avia_routes_url():
        """
        Можно задать диапазон
            /routes/mow/svx/?lang=ru&minDate=2017-08-10&maxDate=2017-08-17
        Или конкретный день
            /routes/mow/svx/moskva-ekaterinburg/?lang=ru&when=2017-08-10
        Или конкретные дни туда-обратно,
        только сейчас страница не отрабатывает как надо
            /routes/mow/svx/?lang=ru&when=2017-08-10&return_date=2017-08-17
        :type min_price: MinPrice
        :rtype: dict
        """
        dt_fmt = '%Y-%m-%d'
        url_args = {'lang': 'ru', 'when': date_fwd.strftime(dt_fmt)}
        if date_bwd:
            url_args['return_date'] = date_bwd.strftime(dt_fmt)
        url_args.update(UTM)
        return 'https://avia.yandex.ru/routes/{from_id}/{to_id}?{query}'.format(
            from_id=point_from['id'],
            to_id=point_to['id'],
            query=urlencode(url_args)
        )

    phrase = (
        u'Из {departure} в {arrival}{one_way}'
        u' за {price} руб. в {month}'
    ).format(
        departure=point_from['new_L_title__ru_genitive'],
        arrival=point_to['new_L_title__ru_accusative'],
        one_way=u' и обратно' if date_bwd else u'',
        price=min_price['price'],
        month=xgettext_month_prepositional(date_fwd.month, 'ru')
    )
    return {
        'id': '{}_{}_{}_{}'.format(
            point_from['id'],
            point_to['id'], date_fwd.month, int(bool(date_bwd))
        ),
        'name': phrase,
        'description': phrase,
        'url': avia_routes_url(),
        'image': IMAGES.url(point_to['id']),
        'from': point_from['new_L_title__ru_nominative'],
        'to': point_to['new_L_title__ru_nominative'],
        'day': date_fwd.day,
        'month': date_fwd.month,
        'year': date_fwd.year,
        'day_backward': date_bwd.day if date_bwd else '',
        'month_backward': date_bwd.month if date_bwd else '',
        'year_backward': date_bwd.year if date_bwd else '',
        'price': min_price['price'],
        'currency': 'RUR',
    }


def build_xml(xml_template_file, params):
    query_xml = loader.render_to_string(xml_template_file, params)
    return ''.join([line.strip() for line in query_xml.splitlines()])


def iter_months(start_date, count=13):
    next_dt = datetime(*start_date.timetuple()[:3])
    for month_count in xrange(count):
        yield next_dt
        next_dt = (next_dt + timedelta(days=31)).replace(day=1)


def query_sets(start_date, months):
    currency_id = Currency.objects.get(code='RUR').id
    months = iter_months(start_date, count=months + 1)
    for left_dt, right_dt in pairwise(months):
        month_prices = MinPrice.objects.filter(
            national_version='ru', date_forward__gte=left_dt,
            date_forward__lt=right_dt, passengers='1_0_0',
            currency_id=currency_id
        ).values('departure_settlement_id', 'arrival_settlement_id', 'price',
                 'date_forward', 'date_backward')
        yield month_prices.filter(date_backward__isnull=True)  # one_way_prices
        yield month_prices.filter(date_backward__isnull=False)  # return_prices


def fill_settlements(min_prices):
    result = []
    settlement_ids = set()
    for mp in min_prices:
        settlement_ids.add(mp['departure_settlement_id'])
        settlement_ids.add(mp['arrival_settlement_id'])

    title_fields = ('new_L_title__ru_genitive', 'new_L_title__ru_accusative',
                    'new_L_title__ru_nominative')
    settlements = {
        s['id']: s
        for s in Settlement.objects.filter(id__in=settlement_ids).select_related(*title_fields).values(*(title_fields + ('id',)))
    }
    for mp in min_prices:
        try:
            mp['departure_settlement'] = settlements[
                mp['departure_settlement_id']
            ]
            mp['arrival_settlement'] = settlements[mp['arrival_settlement_id']]
            result.append(mp)
        except KeyError:
            log.warning('settlements for min price %r not found', mp)
    return result


def get_xml(months, start_date):
    min_prices = []

    def update_min_prices(prices):
        _id = uuid.uuid4()
        log.info('Process %d', _id)
        for _, group in group_by(
            prices,
            key=itemgetter('departure_settlement_id', 'arrival_settlement_id')
        ):
            mp = min(group, key=itemgetter('price'))
            min_prices.append(mp)
        log.info('Done %d', _id)

    p = multiprocessing.Pool(months*2)

    p.map(update_min_prices, query_sets(start_date, months))
    p.close()
    p.join()
    min_prices = fill_settlements(min_prices)
    products = map(product_params, min_prices)
    log.info('Start building XML with %d prices', len(products))
    return build_xml('tradedoubler.xml', {'products': products})


@retry(stop_max_attempt_number=5, wait_fixed=1000)
def upload_to_tradedoubler(xml):
    r = None
    try:
        url = (
            'https://api.tradedoubler.com/1.0/products'
            ';fid={feedId};mode=replace?token={token}'
        ).format(
            feedId=FEED_ID,
            token=TOKEN
        )
        r = requests.post(
            url, data=xml.encode('utf-8'),
            headers={'Content-Type': 'application/xml; charset=utf-8'},
            timeout=20
        )

        r.raise_for_status()
    except Exception as e:
        status = 'Unknown'
        content = 'Unknown'

        if r:
            status = r.status_code
            content = r.content

        log.warning(
            'Can not upload to tradedoubler: [%s] [%s] [%r]',
            status, content, e
        )
        raise

    log.info('Uploading is done. Content: %s', r.content)


def _main(months, start_date):
    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Run only %s allowed.',
                 current_env, allowed_envs_str)
        return

    log.info(u'Start for {} months from {}'.format(months, start_date))

    xml = get_xml(months, start_date)
    upload_to_tradedoubler(xml)
    log.info(u'Done')


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-m', '--months', type='int', dest='months', default=12)
    optparser.add_option('-s', '--start_date', dest='start_date')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    if options.start_date:
        options.start_date = datetime.strptime(options.start_date, '%Y-%m-%d')
    else:
        options.start_date = datetime.now().date()

    create_current_file_run_log()

    script_runner.run_in_process(
        partial(_main, options.months, options.start_date),
        script_code=os.path.basename(__file__)[:-3]
    )
