# -*- coding: utf-8 -*-
from functools import wraps
import cProfile
import os
import pstats
import time

from django.conf import settings

try:
    PROFILING_LOG_BASE = settings.PROFILING_LOG_BASE

except Exception:
    PROFILING_LOG_BASE = None


def profile(func):
    # Не профилировать если не задан путь
    if not PROFILING_LOG_BASE:
        return func

    def decorator(fn):
        profile_file = os.path.join(
            PROFILING_LOG_BASE, fn.__name__ + '.profile'
        )

        if not os.path.exists(PROFILING_LOG_BASE):
            os.makedirs(PROFILING_LOG_BASE)

        @wraps(fn)
        def wrapper(*args, **kwargs):
            # Add a timestamp to the profile output when the callable
            # is actually called.
            base, ext = os.path.splitext(profile_file)

            final_profile_file = (
                base + "-" +
                time.strftime("%Y%m%dT%H%M%S", time.gmtime()) +
                ext
            )

            profiler = cProfile.Profile()

            try:
                result = profiler.runcall(fn, *args, **kwargs)

            finally:
                profiler.dump_stats(final_profile_file)

                with open(base + '.log', 'a') as flowstat_file:
                    p = pstats.Stats(final_profile_file, stream=flowstat_file)
                    p.sort_stats('time', 'cumulative')
                    p.print_stats(12)

            return result

        return wrapper

    return decorator(func)
