# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.contrib.syndication.views import Feed
from django.utils.feedgenerator import Rss201rev2Feed
from django.utils.timezone import get_default_timezone, is_naive, make_aware


class TurboContent(object):
    logo_template = '<figure><img src="{img}" /></figure>'
    header_template = '<header>{logo}<h1>{text}</h1></header>'

    def header(self, header):
        return self.header_template.format(
            logo=self.logo_template.format(img=header['image']) if header['image'] else '',
            text=header['text'],
        )

    def content(self, content):
        return content

    def get(self, header, content):
        return '<![CDATA[{header}{content}]>'.format(
            header=self.header(header),
            content=self.content(content),
        )


class TurboFeedType(Rss201rev2Feed):
    def item_attributes(self, item):
        item_attrs = super(TurboFeedType, self).item_attributes(item)
        item_attrs.update({
            'turbo': 'true',
        })
        return item_attrs

    def rss_attributes(self):
        feed_attrs = super(TurboFeedType, self).rss_attributes()
        feed_attrs.update({
            'xmlns:yandex': 'http://news.yandex.ru',
            'xmlns:media': 'http://search.yahoo.com/mrss/',
            'xmlns:turbo': 'http://turbo.yandex.ru',
        })
        return feed_attrs

    def add_root_elements(self, handler):
        super(TurboFeedType, self).add_root_elements(handler)
        if self.feed['turbo_analytics'] is not None:
            for counter in self.feed['turbo_analytics']:
                handler.startElement('yandex:analytics', counter)
                handler.endElement("yandex:analytics")

    def add_item_elements(self, handler, item):
        super(TurboFeedType, self).add_item_elements(handler, item)

        if item['turbo_topic'] is not None:
            handler.addQuickElement("turbo:topic", item['turbo_topic'])
        if item['turbo_source'] is not None:
            handler.addQuickElement("turbo:source", item['turbo_source'])
        if item['turbo_content'] is not None:
            handler.addQuickElement("turbo:content", item['turbo_content'])
        if item['yandex_related'] is not None:
            handler.addQuickElement("yandex:related", item['yandex_related'])


class TurboFeed(Feed):
    feed_type = TurboFeedType

    def __init__(self):
        super(TurboFeed, self).__init__()
        self.turbo_analytics = []

    def add_analytics(self, counterType, counter, params=''):
        self.turbo_analytics.append({
            'type': counterType,
            'id': str(counter),
            'params': str(params),
        })

    def item_turbo_body(self, item):
        return ''

    def item_turbo_header_text(self, item):
        return self.item_title(item)

    def item_turbo_logo(self, item):
        return self.logo

    def item_turbo_content(self, item):
        return TurboContent().get(
            header={
                'image': self.item_turbo_logo(item),
                'text': self.item_turbo_header_text(item),
            },
            content=self.item_turbo_body(item)
        )

    def item_turbo_source(self, item):
        return self.item_link(item)

    def item_turbo_topic(self, item):
        return self.item_title(item)

    def item_turbo_related(self, item):
        return None

    def item_extra_kwargs(self, item):
        item_extra = super(TurboFeed, self).item_extra_kwargs(item)
        item_extra.update({
            'turbo_topic': self.item_turbo_topic(item),
            'turbo_source': self.item_turbo_source(item),
            'turbo_content': self.item_turbo_content(item),
            'yandex_related': self.item_turbo_related(item),
        })
        return item_extra

    def feed_extra_kwargs(self, obj):
        extra = super(TurboFeed, self).feed_extra_kwargs(obj)
        extra.update({
            'turbo_analytics': self.turbo_analytics,
        })
        return extra

    def _get_dynamic_attr(self, attname, obj, default=None):
        try:
            attr = getattr(self, attname)
        except AttributeError:
            return default
        if callable(attr):
            # Check co_argcount rather than try/excepting the function and
            # catching the TypeError, because something inside the function
            # may raise the TypeError. This technique is more accurate.
            try:
                code = attr.__code__
            except AttributeError:
                code = attr.__call__.__code__
            if code.co_argcount == 2:       # one argument is 'self'
                return attr(obj)
            else:
                return attr()
        return attr

    def get_feed(self, obj):
        """
        Return a feedgenerator.DefaultFeed object, fully populated, for
        this feed. Raise FeedDoesNotExist for invalid parameters.
        """

        link = self._get_dynamic_attr('link', obj)

        feed = self.feed_type(
            title=self._get_dynamic_attr('title', obj),
            subtitle=self._get_dynamic_attr('subtitle', obj),
            link=link,
            description=self._get_dynamic_attr('description', obj),
            language='ru',
            feed_url=link,
            author_name=self._get_dynamic_attr('author_name', obj),
            author_link=self._get_dynamic_attr('author_link', obj),
            author_email=self._get_dynamic_attr('author_email', obj),
            categories=self._get_dynamic_attr('categories', obj),
            feed_copyright=self._get_dynamic_attr('feed_copyright', obj),
            feed_guid=self._get_dynamic_attr('feed_guid', obj),
            ttl=self._get_dynamic_attr('ttl', obj),
            **self.feed_extra_kwargs(obj)
        )

        for item in self._get_dynamic_attr('items', obj):
            title = self._get_dynamic_attr('item_title', item)
            description = self._get_dynamic_attr('item_description', item)
            link = self._get_dynamic_attr('item_link', item)
            enclosures = self._get_dynamic_attr('item_enclosures', item)
            author_name = self._get_dynamic_attr('item_author_name', item)
            if author_name is not None:
                author_email = self._get_dynamic_attr('item_author_email', item)
                author_link = self._get_dynamic_attr('item_author_link', item)
            else:
                author_email = author_link = None

            tz = get_default_timezone()

            pubdate = self._get_dynamic_attr('item_pubdate', item)
            if pubdate and is_naive(pubdate):
                pubdate = make_aware(pubdate, tz)

            updateddate = self._get_dynamic_attr('item_updateddate', item)
            if updateddate and is_naive(updateddate):
                updateddate = make_aware(updateddate, tz)

            feed.add_item(
                title=title,
                link=link,
                description=description,
                unique_id=self._get_dynamic_attr('item_guid', item, link),
                unique_id_is_permalink=self._get_dynamic_attr(
                    'item_guid_is_permalink', item),
                enclosures=enclosures,
                pubdate=pubdate,
                updateddate=updateddate,
                author_name=author_name,
                author_email=author_email,
                author_link=author_link,
                categories=self._get_dynamic_attr('item_categories', item),
                item_copyright=self._get_dynamic_attr('item_copyright', item),
                **self.item_extra_kwargs(item)
            )

        return feed
