# -*- coding: utf-8 -*-
from django.conf import settings
from django.contrib import admin
from django import forms
from django.utils.safestring import mark_safe

from travel.avia.admin.feature_flag_app.models import (
    FeatureFlag, ServiceFeatureFlagRelation, Service
)


st_url_fmt = 'https://st.yandex-team.ru/{}'


class StWidget(forms.TextInput):
    def render(self, name, value, *args, **kwargs):
        html = super(StWidget, self).render(name, value, *args, **kwargs)
        if self.instance.startrack_id:
            html += mark_safe(
                u'\n <a href="{}">Перейти в st</a>'.format(
                    st_url_fmt.format(self.instance.startrack_id)
                )
            )

        return html


class FeatureFlagForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(FeatureFlagForm, self).__init__(*args, **kwargs)
        if hasattr(self, 'instance'):
            self.fields['startrack_id'].widget.instance = self.instance

    class Meta:
        model = FeatureFlag
        widgets = {
            'startrack_id': StWidget,
            'description': forms.Textarea(attrs={'rows': 7, 'cols': 100}),
        }
        exclude = []


class ServiceFeatureFlagRelationInlineForService(admin.TabularInline):
    model = ServiceFeatureFlagRelation

    readonly_fields = ('get_beautiful_enabled',)


class ServiceFeatureFlagRelationInlineForFeatureFlag(admin.TabularInline):
    model = ServiceFeatureFlagRelation


class FeatureFlagAdmin(admin.ModelAdmin):
    form = FeatureFlagForm
    inlines = [
        ServiceFeatureFlagRelationInlineForFeatureFlag,
    ]
    list_display = ('code', 'enabled', 'show_st_url', 'description')
    readonly_fields = ('human_url_view',)

    def show_st_url(self, obj):
        return '<a href="%s">%s</a>' % (
            st_url_fmt.format(obj.startrack_id), obj.startrack_id
        )

    show_st_url.allow_tags = True
    show_st_url.short_description = 'ST'

    class Media:
        css = {
            'all': (
                settings.STATIC_URL + 'feature_flag_app/feature_flag.css',
            )
        }

admin.site.register(FeatureFlag, FeatureFlagAdmin)


class ServiceAdmin(admin.ModelAdmin):
    inlines = [
        ServiceFeatureFlagRelationInlineForService,
    ]

    readonly_fields = ('human_url_view',)

    class Media:
        css = {
            'all': (
                settings.STATIC_URL + 'feature_flag_app/service.css',
            )
        }

admin.site.register(Service, ServiceAdmin)
