# coding=utf-8
from django.core.validators import RegexValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.core.urlresolvers import reverse


class FeatureFlag(models.Model):
    """ Модель функциональных флагов """

    DISABLED = 0
    ENABLED = 1
    USE_AB = 2

    code = models.CharField(
        verbose_name=_(u'код'),
        max_length=50,
        unique=True,
        validators=[RegexValidator(
            ur'^[0-9a-zA-Z-_]+$',
            _(
                u'Код должен удоволетворять ^[0-9a-zA-Z-_]+$ '
            )
        )]
    )

    startrack_id = models.CharField(
        verbose_name=_(u'Номер задачи в startrack'),
        default="",
        null=False,
        blank=True,
        max_length=250,
    )

    description = models.CharField(
        verbose_name=_(u'описание'),
        max_length=250,
    )

    enabled = models.IntegerField(
        verbose_name=_(u'включен?'),
        choices=[
            (0, _(u'выкл')),
            (1, _(u'вкл')),
            (2, _(u'см. AB')),
        ],
        default=0,
    )

    created_at = models.DateTimeField(
        verbose_name=_(u'создан в'),
        auto_now_add=True
    )

    updated_at = models.DateTimeField(
        verbose_name=_(u'обновлен в'),
        auto_now=True,
    )

    def human_url_view(self):
        return u'<a href="{}">Ссылка на эту страницу</a>'.format(
            reverse(
                'feature_flag_app__feature_flag_by_code',
                args=[self.code]
            )
        )

    human_url_view.short_description = u'Удобная ссылка'
    human_url_view.allow_tags = True

    def __repr__(self):
        return u'<{} code={} enabled={}/>'.format(
            self.__class__.__name__,
            self.code,
            self.enabled,
        )

    def __unicode__(self):
        return self.code

    class Meta:
        verbose_name = _(u'функциональный флаг')
        verbose_name_plural = _(u'функциональные флаги')
        app_label = 'feature_flag_app'
        db_table = 'feature_flag_app__feature_flag'


class Service(models.Model):
    """ Модель функциональных флагов """

    code = models.CharField(
        verbose_name=_(u'код'),
        max_length=50,
        unique=True,
        validators=[RegexValidator(
            ur'^[0-9a-zA-Z-_]+$',
            _(
                u'Код должен удоволетворять ^[0-9a-zA-Z-_]+$ '
            )
        )]
    )

    description = models.CharField(
        verbose_name=_(u'описание'),
        max_length=250,
    )

    created_at = models.DateTimeField(
        verbose_name=_(u'создан в'),
        auto_now_add=True,
    )

    def human_url_view(self):
        return u'<a href="{}">Ссылка на эту страницуы</a>'.format(
            reverse(
                'feature_flag_app__service_by_code',
                args=[self.code]
            )
        )

    human_url_view.short_description = u'Удобная ссылка'
    human_url_view.allow_tags = True

    def __repr__(self):
        return u'<{} code={}/>'.format(
            self.__class__.__name__,
            self.code,
        )

    def __unicode__(self):
        return self.code

    class Meta:
        verbose_name = _(u'сервис')
        verbose_name_plural = _(u'сервисы')
        app_label = 'feature_flag_app'
        db_table = 'feature_flag_app__service'


class ServiceFeatureFlagRelation(models.Model):
    """ Модель, связывающая сервис и функциональный флаг"""
    service = models.ForeignKey(
        Service,
        verbose_name=_(u'сервис'),
        on_delete=models.CASCADE
    )

    feature_flag = models.ForeignKey(
        FeatureFlag,
        verbose_name=_(u'функциональный флаг'),
        on_delete=models.CASCADE
    )

    created_at = models.DateTimeField(
        verbose_name=_(u'создан в'),
        auto_now_add=True,
    )

    def get_beautiful_enabled(self):
        return {
            FeatureFlag.DISABLED: u'🚫',
            FeatureFlag.ENABLED: u'✅',
            FeatureFlag.USE_AB: u'Use AB',
        }.get(self.feature_flag.enabled, u'Unknown')

    get_beautiful_enabled.short_description = _(u"Включен?")

    def __repr__(self):
        return u'<{} service={} feature_flag={}/>'.format(
            self.__class__.__name__,
            self.service_id,
            self.feature_flag_id,
        )

    class Meta:
        unique_together = ('service', 'feature_flag')
        verbose_name = _(u'связь сервиса и функциональных флага')
        verbose_name_plural = _(u'связи сервисов и функциональных флагов')
        app_label = 'feature_flag_app'
        db_table = 'feature_flag_app__service_feature_flag_relation'
