# -*- coding: utf-8 -*-
""" Поставщик различной информации """

import os

from ylog.utils import log_warning

from django.conf import settings
from travel.avia.library.python.common.utils.caching import cached

try:
    import geobase6

    try:
        lookup = geobase6.Lookup("/var/cache/geobase/geodata6.bin")
    except RuntimeError:
        lookup = geobase6.Lookup(os.path.expanduser("~/.local/var/cache/geodata6.bin"))

except (ImportError, RuntimeError):
    log_warning()
    lookup = None

from travel.avia.library.python.common.utils.mysql_try_hard import mysql_try_hard
from travel.avia.library.python.common.models.geo import Settlement, Region, Country


@mysql_try_hard
def get_city_from_regions(regions):
    # Пробуем найти известный нам объект

    for r in regions:
        if r.type in [6, 7]:  # Город, село
            try:
                # город
                settlement = Settlement.objects.get(_geo_id=r.id)
                if not settlement.hidden:
                    return settlement

                # столица области
                if settlement.region:
                    try:
                        return Settlement.hidden_manager.get(region=settlement.region, majority=2)
                    except Settlement.DoesNotExist:
                        pass

                # столица страны
                try:
                    return Settlement.hidden_manager.get(country=settlement.country, majority=1)
                except Settlement.DoesNotExist:
                    pass

            except Settlement.DoesNotExist:
                pass

            # Пробуем найти по названию
            try:
                return Settlement.hidden_manager.filter(title=r.name.decode('utf-8'))[0]
            except IndexError:  # Если нет такого города в базе
                pass

        elif r.type == 5:  # субъект федерации<!--  (края, автономные округа, области, республики, а также любые части иностранных государств (Крым, Гонконг и т.п.)) -->
            # Пробуем найти столицу области
            try:
                region = Region.hidden_manager.get(_geo_id=r.id)
                return region.get_capital()
            except Region.DoesNotExist:
                pass

        elif r.type == 3:  # Страна
            # Пробуем найти столицу страны
            try:
                country = Country.objects.get(_geo_id=r.id)
                return country.get_capital()
            except Country.DoesNotExist:
                pass

    return None


@cached(lambda ip: "/geoip/" + ip, timeout=settings.CACHES['default']['LONG_TIMEOUT'])
def get_city_by_ip(ip):
    if lookup:
        try:
            regions = lookup.regions(str(ip))

            geoid = regions[0].id

            city = get_city_from_regions(regions)

            return city, geoid

        except Exception:
            # Не падать, писать ошибки
            log_warning()

    return None, None


def get_city(request, ip):
    """ Определяет город клиента по ip
    Вход: request, ip
    Выход: экземпляр models.Settlement """

    if request.GET.get('forced_city_id'):
        try:
            settlement = Settlement.hidden_manager.get(id=request.GET.get('forced_city_id'))
            return settlement, settlement._geo_id
        except Settlement.DoesNotExist:
            pass

    # Локальная кука pda.rasp.yandex.ru
    if 'rasp_tablo' in request.COOKIES:
        try:
            settlement = Settlement.hidden_manager.get(pk=int(request.COOKIES['rasp_tablo']))
            return settlement, settlement._geo_id
        except (IndexError, ValueError, Settlement.DoesNotExist):
            pass

    gid = request.GET.get('geo') or request.COOKIES.get('yandex_gid')

    if gid and lookup:
        try:
            gid = int(gid)

            try:
                settlement = get_city_from_regions([lookup.region_by_id(gid)])

                return settlement, gid
            except (ValueError, RuntimeError, OverflowError):
                try:
                    open(settings.EXCEPTION_LOG_FILE, 'a').write('Cookie with code "%s" found. Settlement with such geo_id is absent in our db.\n' % gid)
                except IOError:
                    pass

        except ValueError:
            pass

    user_agent = request.META.get('HTTP_USER_AGENT')

    # RASP-12531
    if user_agent:
        for bot_token in [
            'YandexBot', 'YandexFavicons',
            'Googlebot',
            'bingbot',
        ]:
            if bot_token in user_agent:
                return None, None

    # Отладочные варианты
    # ip = '87.250.249.36' # Москва
    # ip = '194.226.146.5' # Екатеринбург
    # ip = '80.78.117.59' # Серов
    # ip = '195.58.238.156' # Донецк (Украина)
    # ip = '84.201.230.121' # Минск
    # ip = '85.64.35.189' # Хайфа
    # ip = '183.64.35.189'
    # ip = '85.143.0.30' # Нижний Новгород

    if ip:
        city, gid = get_city_by_ip(ip)

        if city:
            return city, gid

    # Если город не определен, в middleware выставляется Москва
    return None, None
