# -*- coding: utf-8

from django import forms
from django.conf import settings
from django.contrib.admin.widgets import AdminDateWidget
from django.contrib.auth.decorators import login_required
from django.shortcuts import render_to_response as django_render_to_response
from django.template import RequestContext
from django.utils.safestring import mark_safe
from django.utils.html import escape
from django.utils.text import Truncator

from travel.avia.admin.leaflets.views import leaflet
from travel.avia.library.python.common.models.geo import ExternalDirection, Station


class RawIdWidget(forms.TextInput):
    """
    A Widget for displaying ForeignKeys in the "raw_id" interface rather than
    in a <select> box.
    """
    def __init__(self, model, attrs=None):
        self.model = model
        super(RawIdWidget, self).__init__(attrs)

    def render(self, name, value, attrs=None):
        if attrs is None:
            attrs = {}
        related_url = '/admin/%s/%s/' % (self.model._meta.app_label,
                                           self.model._meta.object_name.lower())
        params = self.url_parameters()
        if params:
            url = '?' + '&amp;'.join(['%s=%s' % (k, v) for k, v in params.items()])
        else:
            url = ''
        if 'class' not in attrs:
            attrs['class'] = 'vForeignKeyRawIdAdminField'  # The JavaScript looks for this hook.
        output = [super(RawIdWidget, self).render(name, value, attrs)]
        # TODO: "id_" is hard-coded here. This should instead use the correct
        # API to determine the ID dynamically.
        output.append('<a href="%s%s" class="related-lookup" id="lookup_id_%s" onclick="return showRelatedObjectLookupPopup(this);"> ' % (related_url, url, name))
        output.append('<img src="%simg/admin/selector-search.gif" width="16" height="16" alt="%s" /></a>' % (settings.ADMIN_MEDIA_PREFIX, 'Lookup'))
        if value:
            output.append(self.label_for_value(value))
        return mark_safe(u''.join(output))

    def base_url_parameters(self):
        params = {}
        return params

    def url_parameters(self):
        from django.contrib.admin.views.main import TO_FIELD_VAR
        params = self.base_url_parameters()
        params.update({TO_FIELD_VAR: "pk"})
        return params

    def label_for_value(self, value):
        key = "pk"
        obj = self.model._default_manager.get(**{key: value})
        return '&nbsp;<strong>%s</strong>' % escape(Truncator(obj).words(14))


class ScheduleLeafletForm(forms.Form):
    external_direction = forms.ModelChoiceField(queryset=ExternalDirection.objects.all(),
                                                required=False)
    station1 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station2 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station3 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station4 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station5 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station6 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    date1 = forms.DateField(required=False, widget=AdminDateWidget)
    date2 = forms.DateField(required=False, widget=AdminDateWidget)
    date3 = forms.DateField(required=False, widget=AdminDateWidget)
    date4 = forms.DateField(required=False, widget=AdminDateWidget)
    date5 = forms.DateField(required=False, widget=AdminDateWidget)
    date6 = forms.DateField(required=False, widget=AdminDateWidget)

    def clean(self):
        stations = []
        dates = []
        for n in range(1, 7):
            if self.cleaned_data.get('station%s' % n):
                stations.append(self.cleaned_data.get('station%s' % n))
        for n in range(1, 7):
            if self.cleaned_data.get('date%s' % n):
                dates.append(self.cleaned_data.get('date%s' % n))
        if not dates:
            raise forms.ValidationError(u"Нужно выбрать хотябы одну дату")
        if not self.cleaned_data['external_direction'] and not stations:
            raise forms.ValidationError(u"Нужно выбрать хотя бы одну станцию или внешнее направление")
        elif self.cleaned_data['external_direction'] and stations:
            raise forms.ValidationError(u"Нужно выбрирать либо станцию, либо направления")

        self.cleaned_data['dates'] = dates
        self.cleaned_data['stations'] = stations
        return self.cleaned_data


@login_required
def schedule_leaflets(request):
    if request.POST:
        form = ScheduleLeafletForm(request.POST)
        if form.is_valid():
            return leaflet(form.cleaned_data['external_direction'],
                           form.cleaned_data['stations'], form.cleaned_data['dates'])

    else:
        form = ScheduleLeafletForm()

    context = {
        'title': u'Генератор листовок',
        'form': form,
        'media': form.media,
        }

    return django_render_to_response('admin/leaflets/leaflet.html', context, RequestContext(request))
