# -*- coding: utf-8 -*-

import re
from datetime import date, datetime, timedelta

from django.http import HttpResponse

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.schedule import RThread, RTStation
from travel.avia.library.python.common.utils.date import human_date_without_year


def leaflet(direction, stations, dates):
    if direction:
        stations = direction.stations.all()

    today = date.today()

    lines = []

    for station in stations:
        station_lines = list(process_station(today, station, dates))

        if station_lines:
            lines.append(station.title)

            for line in station_lines:
                lines.append("    " + line)

    return HttpResponse("\n".join(lines), content_type='text/plain')


def calc_time(day, start_time, delta):
    if delta is None:
        return None

    return datetime.combine(day, start_time) + timedelta(minutes=delta)


class Update(object):
    def __init__(self, thread, path):
        self.thread = thread
        self.path = path


def process_station(today, station, dates):
    # Выбираем все нитки, проходящие через данную станцию
    threads = list(RThread.objects.filter(
        rtstation__station=station, route__t_type__code='suburban'
    ).only('id', 'basic_thread__id'))

    # id всех идущих ниток
    threads_ids = set(t.id for t in threads)

    # Добавляем id базовых для всех идущих
    threads_ids.update(t.basic_thread_id for t in threads if t.basic_thread_id)

    # Добавляем id обновлений для всех идущих базовых и базовых для идущих обновлений
    threads_ids.update(RThread.objects.filter(basic_thread__id__in=list(threads_ids)).values_list('id', flat=True))

    # Вытаскиваем все треды
    threads = list(RThread.objects.filter(id__in=threads_ids))

    basic_threads = [t for t in threads if t.type.code == 'basic']

    update_threads_by_basic = {}

    for t in threads:
        if t.type.code == 'basic':
            continue

        update_threads_by_basic.setdefault(t.basic_thread_id, []).append(t)

    # Вытаскиваем все rtstations
    rtstations_by_thread = {}
    paths_by_thread = {}

    for rts in RTStation.objects.filter(thread__in=threads).select_related('station').order_by('id'):
        if rts.station == station:
            rtstations_by_thread[rts.thread_id] = rts

        paths_by_thread.setdefault(rts.thread_id, []).append(rts)

    lines_by_date = {}

    for basic_thread in basic_threads:
        update_threads = update_threads_by_basic.get(basic_thread.id, [])

        basic_rtstation = rtstations_by_thread.get(basic_thread.id)

        updates_by_date = {}

        # Какие треды в какой день посещают эту станцию
        for thread in update_threads:
            rtstation = rtstations_by_thread.get(thread.id)

            update_path = paths_by_thread[thread.id]

            for day in thread.get_mask(today).dates():
                update = Update(thread, update_path)

                if not rtstation and not basic_rtstation:
                    continue

                update.last_arrival = calc_time(day, thread.start_time, update.path[-1].arrival)

                if rtstation is not None:
                    update.platform = rtstation.L_platform()

                    update.departure = calc_time(day, thread.start_time, rtstation.departure)
                    update.arrival = calc_time(day, thread.start_time, rtstation.arrival)
                else:
                    update.cancel_station = True

                if basic_rtstation:
                    update.basic_departure = calc_time(day, basic_thread.start_time, basic_rtstation.departure)
                    update.basic_arrival = calc_time(day, basic_thread.start_time, basic_rtstation.arrival)
                else:
                    update.basic_departure = None
                    update.basic_arrival = None

                station_date = (update.basic_departure or update.basic_arrival or update.departure or update.arrival).date()

                updates_by_date.setdefault(station_date, []).append(update)

        for date_ in dates:
            try:
                updates = updates_by_date[date_]
            except KeyError:
                continue

            lines = lines_by_date.setdefault(date_, [])

            for update in updates:
                chunks = process_thread(station, basic_thread, paths_by_thread.get(basic_thread.id, []), update)

                station_text = chunks.next()

                text = " ".join(chunks)

                text = re.sub(r'- \.', '-', text)
                text = re.sub(r'( |^)[.-]$', '', text)
                text = re.sub(r' \. (.)', lambda m: '. ' + m.group(1).upper(), text)
                text = re.sub(r'&nbsp;', ' ', text)

                if text:
                    lines.append("%s %s %s" % (basic_thread.number, station_text, text))

    for date_ in dates:
        try:
            lines = lines_by_date[date_]
        except KeyError:
            continue

        yield human_date_without_year(date_)

        for line in lines:
            yield '    ' + line


def format_platform(platform, case):
    return platform


def format_time(t):
    if t:
        return t.strftime("%H:%M")
    else:
        return ""


def process_thread(station, basic_thread, basic_path, update):
    update_stations = [r.station for r in update.path]

    basic_first_station = basic_path[0].station
    basic_last_station = basic_path[-1].station

    update_first_station = update.path[0].station
    update_last_station = update.path[-1].station

    basic_rtstation = (r for r in basic_path if r.station == station).next()

    basic_platform = basic_rtstation.L_platform()

    basic_stops = [r.station for r in basic_path if r.stop]
    update_stops = [r.station for r in update.path if r.stop]

    if station == basic_last_station:
        yield u'из %s %s' % (basic_first_station.title_ru_genitive, format_time(update.basic_arrival))
    else:
        yield "%s %s" % (basic_last_station.title, format_time(update.basic_departure))

    yield '-'

    # Вариант 2
    if update.thread.type.code == 'cancel' or station not in update_stations:
        yield u'ОТМЕНЕН'
        return

    # вариант 3
    if station == basic_last_station == update_last_station:
        last_changed = basic_first_station != update_first_station

        time_changed = update.arrival != update.basic_arrival

        if last_changed or time_changed:
            yield u'прибудет'

            if last_changed:
                yield update_first_station.phrase_from

            if time_changed:
                yield u'в %s' % format_time(update.arrival)

            return

    # вариант 4
    if station not in basic_stops and station in update_stops:
        yield u'НАЗНАЧАЕТСЯ'

        if update.platform:
            yield u'(с %s)' % format_platform(update.platform, 'gen')

        return

    # Вариант 1
    if station != basic_last_station and station == update_last_station:
        yield u'ОТМЕНЕН (следует только до %s)' % update_last_station.title_gen
        return

    # Если рейс останавливался на этой станций, а сейчас не останавливается, но проходит
    if station in basic_stops and station not in update_stops and station in update_stations:
        yield u'ОТМЕНЕН (проследует без остановки)'
        return

    pass_blocks = []

    if update.departure != update.basic_departure:
        pass_blocks.append(u'в %s' % format_time(update.departure))

    if basic_last_station != update_last_station:
        pass_blocks.append(u"до %s (приб %s)" % (
            update_last_station.title_gen,
            format_time(update.last_arrival),
            ))

    if update.platform and basic_platform != update.platform:
        pass_blocks.append(u'(по %s)' % format_platform(update.platform, 'dat'))

    if pass_blocks:
        yield u'проследует'

        for block in pass_blocks:
            yield block

    yield '.'

    cancelled_stops = [s for s in basic_stops if s not in update_stops and s in update_stations]
    new_stops = [s for s in update_stops if s not in basic_stops]

    if cancelled_stops:
        yield u'отменены остановки: %s.' % ', '.join(station.title for station in cancelled_stops)

    if new_stops:
        yield u'назначены остановки: %s.' % ', '.join(station.title for station in new_stops)


def test():
    station = Station.objects.get(id=2000001)

    today = date.today()

    dates = [date(2010, 04, 21)]

    for line in process_station(today, station, dates):
        print line
