# -*- coding: utf-8 -*-

from StringIO import StringIO
from email.header import Header

from functools import update_wrapper, partial
from operator import isCallable
from traceback import format_exc

from django import forms
from django.conf import settings
from django.contrib.admin import ModelAdmin
from django.contrib.admin.utils import lookup_needs_distinct
from django.contrib.admin.views.main import ChangeList, ORDER_VAR
from django.db import models
from django.db.models.fields.related import RelatedField
from django.http import HttpResponse
from django.utils.encoding import smart_str

from travel.avia.admin.avia.sync_rasp.import_rasp_info_schema import FK, FILE, NEW_RASP_SYNC_MODELS, IMPORT_MODELS
from travel.avia.library.python.common.utils.date import get_time_zone_choices
from travel.avia.library.python.common.utils.unicode_csv import UnicodeWriter


FORMAT_PARAM = '_format'
COLUMNS_PARAM = '_columns'
ENCODING_PARAM = '_encoding'
LIMIT_PARAM = '_limit'
BOOLEAN_FORMAT = '_boolean_format'
DEFAULT_BOOLEAN_FORMAT = 'int'
IGNORED_PARAMS = (FORMAT_PARAM, COLUMNS_PARAM, ENCODING_PARAM, LIMIT_PARAM, BOOLEAN_FORMAT)
DEFAULT_FORMAT = 'csv'
DEFAULT_ENCODING = 'utf-8'


class AllLookupModelAdmin(ModelAdmin):
    save_on_top = True

    def lookup_allowed(self, lookup, value):
        return True

    def render_change_form(self, request, context, add=False, change=False, form_url='', obj=None):
        context.update({
            'need_topic': len(self.get_fieldsets(request, obj)) > 1
        })

        return super(AllLookupModelAdmin, self).render_change_form(request, context, add, change,
                                                                   form_url, obj)


class ChangeListNoDefaultOrdering(ChangeList):
    def get_ordering(self, request, queryset):
        params = self.params
        ordering = list(self.model_admin.get_ordering(request)
                        or self._get_default_ordering())
        if ORDER_VAR in params:
            # Clear ordering and used params
            ordering = []
            order_params = params[ORDER_VAR].split('.')
            for p in order_params:
                try:
                    none, pfx, idx = p.rpartition('-')
                    field_name = self.list_display[int(idx)]
                    order_field = self.get_ordering_field(field_name)
                    if not order_field:
                        continue  # No 'admin_order_field', skip it
                    ordering.append(pfx + order_field)
                except (IndexError, ValueError):
                    continue  # Invalid ordering specified, skip it.

        # Add the given query's ordering fields, if any.
        ordering.extend(queryset.query.order_by)

        return ordering


class RaspExportModelAdmin(AllLookupModelAdmin):
    def get_readonly_fields(self, request, obj=None):
        readonly_fields = set(super(RaspExportModelAdmin, self).get_readonly_fields(request, obj))
        if not settings.ALLOW_EDIT_SYNC_WITH_RASP_FIELDS:
            if self.model in NEW_RASP_SYNC_MODELS:
                fields = []
                for im in IMPORT_MODELS:
                    if im['model'] == self.model:
                        fields = im['fields']
                for field in fields:
                    if isinstance(field, (FK, FILE)):
                        readonly_fields.add(field.field)
                    else:
                        if field.startswith('new_L'):
                            # skip new translations here, they will be handled in TranslationsMixin
                            pass
                        else:
                            readonly_fields.add(field)
        return list(readonly_fields)

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(RaspExportModelAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^export/$', wrap(self.export_view), name='%s_%s_export' % info),
        ] + urlpatterns

        return urlpatterns

    def get_export_ignored_params(self):
        return IGNORED_PARAMS

    def export_view(self, request):
        try:
            self.request = request

            query_set = self.get_export_query_set()

            columns = self.get_columns()

            rows = self.get_rows(query_set, columns)

            formatter = self.get_export_formatter()

            return formatter(columns, rows)
        except Exception:
            return HttpResponse(u'Ошибка запроса, проверьте параметры:\n{}'.format(format_exc()),
                                content_type='text/plain; charset=utf-8')

    def get_export_query_set(self):
        ignored_params = self.get_export_ignored_params()

        lookup_params = self.request.GET.copy()

        for ignored in ignored_params:
            if ignored in lookup_params:
                del lookup_params[ignored]

        # Normalize the types of keys
        for key, value in lookup_params.items():
            if not isinstance(key, str):
                # 'key' will be used as a keyword argument later, so Python
                # requires it to be a string.
                del lookup_params[key]
                lookup_params[smart_str(key)] = value

        use_distinct = False

        filter_params = {}

        add_filters = []

        for key, value in lookup_params.items():
            filter_param = prepare_lookup_value(key, value)

            if isinstance(filter_param, models.Q):
                add_filters.append(filter_param)
            else:
                filter_params[key] = filter_param

            use_distinct = (use_distinct or lookup_needs_distinct(self.opts, key))

        query_set = self.get_queryset(self.request).filter(**filter_params)

        for add_filter in add_filters:
            query_set = query_set.filter(add_filter)

        return query_set

    def get_columns(self):
        fields = filter(None, self.request.GET.get(COLUMNS_PARAM, u"").split(u","))

        return fields or [f.column for f in self.opts.local_fields]

    def get_values(self, obj, columns):
        return [self.get_column_value(obj, column) for column in columns]

    def get_column_value(self, obj, column):
        current_value = obj
        for attr_name in column.split('__'):
            current_value = self.get_attr_or_field_value(current_value, attr_name)

        return self.prepare_to_export(current_value)

    def get_attr_or_field_value(self, obj, attr_name):
        attr = getattr(obj, attr_name)
        if isCallable(attr):
            return attr()
        else:
            return attr

    def get_rows(self, query_set, columns):
        limit = self.request.GET.get(LIMIT_PARAM, None)
        if limit:
            query_set = query_set[:int(limit)]

        for obj in query_set:
            yield self.get_values(obj, columns)

    def prepare_to_export(self, value):
        if value is None:
            return u""
        elif isinstance(value, models.Model):
            return unicode(value.id)
        elif isinstance(value, bool):
            if self.request.GET.get(BOOLEAN_FORMAT, DEFAULT_BOOLEAN_FORMAT) == 'int':
                return unicode(int(value))
            else:
                return unicode(value)
        else:
            return unicode(value)

    def get_export_formatter(self):
        format = self.request.GET.get(FORMAT_PARAM, DEFAULT_FORMAT)

        encoding = self.request.GET.get(ENCODING_PARAM, DEFAULT_ENCODING)

        return partial(getattr(self, "%s_export_formatter" % format), encoding=encoding)

    def csv_export_formatter(self, columns, rows, encoding, delimiter=';', content_type="text/csv", ext="csv"):
        stream = StringIO()
        writer = UnicodeWriter(stream, encoding=encoding, delimiter=delimiter)

        def content():
            writer.writerow(columns)
            yield stream.getvalue()

            stream.truncate(0)

            for row in rows:
                writer.writerow(row)

                yield stream.getvalue()

                stream.truncate(0)

        response = HttpResponse(''.join(content()), content_type="%s; charset=%s" % (content_type, encoding))

        filename = self.model.__name__.lower() + 's.' + ext

        response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(filename, encoding))

        return response

    def tsv_export_formatter(self, columns, rows, encoding):
        return self.csv_export_formatter(columns, rows, encoding, delimiter="\t", ext="tsv")


def prepare_lookup_value(key, value):
    """
    Returns a lookup value prepared to be used in queryset filtering.
    """
    # if key ends with __in, split parameter into separate values
    if key.endswith('__in'):
        value = value.split(',')
    # if key ends with __isnull, special case '' and false
    if key.endswith('__isnull'):
        if value.lower() in ('', '0', 'false'):
            value = False
        else:
            value = True

    if key.endswith('__empty'):
        key = key.replace('__empty', "")

        is_empty = models.Q(**{key: ""}) | models.Q(**{key + '__isnull': True})

        if value.lower() in ('', '0', 'false'):
            return ~ is_empty
        else:
            return is_empty

    return value


class OrderingOnlyOnSmallQuerysetChangeList(ChangeList):
    """
    Класс предназначен для запрета сортировки больших выборок.

    MAX_ROWS_FOR_ORDERING - при привышении данного количества
    записей никакая сортировка производится не будет.

    Использовать можно в потомках ModelAdmin примерно так:

    def get_changelist(self, request, **kwargs):
        change_list_class = OrderingOnlyOnSmallQuerysetChangeList
        change_list_class.MAX_ROWS_FOR_ORDERING = 500

        return change_list_class
    """

    MAX_ROWS_FOR_ORDERING = 5000

    def get_query_set(self, request):
        queryset = super(OrderingOnlyOnSmallQuerysetChangeList, self).get_query_set(request)

        if queryset.order_by().count() > self.MAX_ROWS_FOR_ORDERING:
            queryset = queryset.order_by()

        return queryset


class OrderingOnlyOnSmallQuerysetModelAdminMixin(object):
    """
    Класс предназначен для запрета сортировки больших выборок.

    MAX_ROWS_FOR_ORDERING - при привышении данного количества
    записей никакая сортировка производится не будет.

    Использование:
    class MyAdmin(OrderingOnlyOnSmallQuerysetModelAdminMixin, ModelAdmin):
        MAX_ROWS_FOR_ORDERING = 100

    """

    MAX_ROWS_FOR_ORDERING = 5000

    def get_changelist(self, request, **kwargs):
        change_list_class = OrderingOnlyOnSmallQuerysetChangeList
        change_list_class.MAX_ROWS_FOR_ORDERING = self.MAX_ROWS_FOR_ORDERING

        return change_list_class


def model_form_time_zone_mixin(*fields, **kwargs):
    additonal_choices = kwargs.get('additonal_choices', [])
    class_attrs = {
        f: forms.ChoiceField(required=kwargs.get('required', True), choices=additonal_choices + get_time_zone_choices(), initial='')
        for f in fields
    }

    return type('TimezoneMixin', (forms.ModelForm,), class_attrs)

UPDATE_FIELDS = {
    'TranslatedTitle': (
        'ru_nominative', 'ru_genitive', 'ru_dative', 'ru_accusative',
        'ru_locative', 'en_nominative', 'de_nominative', 'tr_nominative',
        'uk_nominative', 'uk_accusative'
    ),
    'TranslatedText': ('ru', 'en', 'de', 'tr', 'uk')
}


class TranslationsMixin(object):
    def _get_related_models(self):
        related_models = filter(
            lambda x: isinstance(x, RelatedField) and
                      x.related_model._meta.object_name in UPDATE_FIELDS,
            self.model._meta.local_fields)

        return [(model.name, model.related_model._meta.object_name)
                for model in related_models]

    def _get_related_readonly_fields(self):
        if settings.ALLOW_EDIT_SYNC_WITH_RASP_FIELDS:
            return set()

        related_readonly_fields = set()
        if self.model in NEW_RASP_SYNC_MODELS:
            fields = []
            for im in IMPORT_MODELS:
                if im['model'] == self.model:
                    fields = im['fields']
            for field in fields:
                if not isinstance(field, (FK, FILE)) and field.startswith('new_L'):
                    related_readonly_fields.add(field)
        return related_readonly_fields

    def get_form(self, request, obj=None, **kwargs):
        related_readonly_fields = self._get_related_readonly_fields()
        related_models = self._get_related_models()
        if related_models:
            class FormWithTranslationsFields(self.form):
                attrs = locals()
                for field, related_instace_name in related_models:

                    for tr_field in UPDATE_FIELDS[related_instace_name]:
                        field_name = "{}_{}".format(field, tr_field)
                        readonly = field_name in related_readonly_fields
                        if readonly:
                            widget = forms.Textarea(attrs={'colls': 60, 'rows': 1, 'disabled': True})
                        else:
                            widget = forms.Textarea(attrs={'colls': 60, 'rows': 1})
                        attrs[field_name] = forms.Field(
                            widget=widget,
                            required=False,
                            initial=getattr(getattr(obj, field), tr_field) if obj else ''
                        )

                def save(self, *args, **kwargs):
                    for field, related_instace_name in related_models:
                        related_instance = getattr(self.instance, field, None)
                        related_info = {tr_field: self.cleaned_data[
                            "{}_{}".format(field, tr_field)] for tr_field in UPDATE_FIELDS[related_instace_name]}
                        if not related_instance:
                            related_instance = globals()[related_instace_name](**related_info)
                        else:
                            for tr_field, value in related_info.iteritems():
                                setattr(related_instance, tr_field, value)
                        related_instance.save()
                        setattr(self.instance, field, related_instance)
                    result = super(FormWithTranslationsFields, self).save(*args, **kwargs)

                    return result
            self.form = FormWithTranslationsFields
        return super(TranslationsMixin, self).get_form(request, obj, **kwargs)

    def get_fieldsets(self, request, obj=None):
        """

        """
        related_models = self._get_related_models()
        if self.fieldsets is not None and related_models:
            for title, options in self.fieldsets:
                new_options = list(options['fields'])
                for field in options['fields']:
                    for rel_field, related_instace_name in related_models:
                        if field == rel_field:
                            new_options.remove(field)
                            new_options.extend(
                                map(lambda x: '{}_{}'.format(field, x),
                                    UPDATE_FIELDS[related_instace_name]))
                options['fields'] = new_options

        return super(TranslationsMixin, self).get_fieldsets(request, obj)
