# coding=utf-8
from collections import defaultdict
from logging import Logger, getLogger

from django.conf import settings
from django.db import transaction

from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.iata_correction import IATACorrector

from travel.avia.admin.lib.flight_frequency_collector import FlightFrequencyCollector, flight_frequency_collector
from travel.avia.admin.lib.yt_helpers import YtTableProvider, yt_table_provider


class AirlinePopularScoreUpdater(object):
    """
    Обновлялка очков популярности авиакомпании
    Популярность авиакомпании считается следующем ообразом:
    1) берем все номера рейсов, которые учавствовали в покупке
    2) считаем популярность рейсов
    3) затем по номеру рейса определяем АК
    4) затем обновляем очки популярности. Одно очко - одно участие самолета в перелете
    """
    def __init__(self, flight_frequency_collector, yt_table_provider, iata_corrector, logger):
        # type: (FlightFrequencyCollector, YtTableProvider, IATACorrector, Logger) -> None
        self._flight_frequency_collector = flight_frequency_collector
        self._yt_table_provider = yt_table_provider
        self._iata_corrector = iata_corrector
        self._logger = logger

    def update(self, precise):
        # type: (int) -> None
        self._logger.info('Start')
        sources = self._yt_table_provider.get_last(
            path='//home/rasp/logs/rasp-popular-flights-log',
            days=precise
        )

        self._logger.info('input sources: %r', sources)
        self._logger.info('start collect flight frequency')
        number_with_frequencies_list = self._flight_frequency_collector.collect_to_list(sources)
        self._logger.info('end collect flight frequency. count uniq flights %d', len(number_with_frequencies_list))

        self._logger.info('start build company_to_frequency map')
        company_to_frequency = self._build_company_to_frequency(
            number_with_frequencies_list
        )
        self._logger.info('end build company_to_frequency map')

        self._logger.info('start update popular score in databases')
        self._update_popular_scores(company_to_frequency)
        self._logger.info('end update popular score in databases')

        self._logger.info('End')

    def _build_company_to_frequency(self, number_with_frequencies_list):
        company_to_frequency_by_national_version = defaultdict(
            lambda: {nv: 0 for nv in settings.AVIA_NATIONAL_VERSIONS}
        )

        company_ids_by_flight_number = self._iata_corrector.flight_numbers_to_carriers(
            set([data.get('number').decode('utf-8') for data in number_with_frequencies_list])
        )

        for data in number_with_frequencies_list:
            number = data.pop('number').decode('utf-8')

            company_id = company_ids_by_flight_number.get(number)
            if not company_id:
                continue
            try:
                company = Company.objects.get(id=company_id)
            except Company.DoesNotExist:
                self._logger.warning('Company (id=%d) not found', company_id)
                continue

            frequency_by_national_version = company_to_frequency_by_national_version[company]
            national_version_set = set(settings.AVIA_NATIONAL_VERSIONS)
            for national_version, count in data.iteritems():
                if national_version not in national_version_set:
                    self._logger.info('can not find national_version: %s', number)
                    continue
                frequency_by_national_version[national_version] += count

        return company_to_frequency_by_national_version

    def _update_popular_scores(self, company_to_frequency):
        with transaction.atomic():
            for company in Company.objects.filter(t_type_id=TransportType.PLANE_ID):
                frequency_by_national_version = company_to_frequency[company]

                count = 0
                for national_version in frequency_by_national_version:
                    setattr(
                        company,
                        'popular_score_for_{}'.format(national_version),
                        frequency_by_national_version[national_version]
                    )
                    count += frequency_by_national_version[national_version]

                company.popular_score = count
                company.save()


_logger = getLogger(__name__)

airline_popular_score_updater = AirlinePopularScoreUpdater(
    flight_frequency_collector=flight_frequency_collector,
    yt_table_provider=yt_table_provider,
    iata_corrector=IATACorrector(logger=_logger),
    logger=_logger,
)
