# -*- coding: utf8 -*-
import logging
from typing import List, Optional, Tuple
from datetime import datetime

from django.conf import settings

from travel.avia.library.python.balance_client.client import Client as BalanceClient, BALANCE_AVIA_SERVICE_ID
from travel.avia.library.python.common.models.partner import UpdateHistoryRecord, Partner

from travel.avia.admin.lib.partner_mail_sender import (
    send_mail,
    LOW_BALANCE_NOTIFICATION_MAIL_TYPE,
    NONPOSTIVE_BALANCE_NOTIFICATION_MAIL_TYPE,
    POSTIVE_BALANCE_NOTIFICATION_MAIL_TYPE
)

FORMAT = u"%Y-%m-%d %H:%M"
DEFAULT_NOTIFY_BALANCE_THRESHOLD = 300

log = logging.getLogger(__name__)


def _update_balance(partner, current_balance):
    # type: (Partner, float) -> Optional[str]
    """
    Обновляем текущий баланс для одного партнера.
    И в случае необходимости определяемся с темой письма для оповещения
    состояния лицевого счета партнера
    """
    old_balance = partner.current_balance
    partner.current_balance = current_balance
    partner.balance_updated_at = datetime.utcnow()

    log.info('update balance old: [%r] new: [%r]', old_balance, current_balance)

    if partner.current_balance <= 0 and not partner.disabled:
        if partner.enabled_with_negative_balance:
            return None

        partner.disable(yandex_login='balance', role='admin')
        return NONPOSTIVE_BALANCE_NOTIFICATION_MAIL_TYPE

    if old_balance <= 0 < partner.current_balance and partner.disabled:
        last_update = (
            partner.history_update_records
                   .filter(action=UpdateHistoryRecord.CHANGE_ACTIVE_STATUS_ACTION)
                   .order_by('time')
                   .last()
        )
        if (
            last_update
            and last_update.updater_yandex_login == 'balance'
        ):
            partner.enable(yandex_login='balance', role='admin')
            return POSTIVE_BALANCE_NOTIFICATION_MAIL_TYPE

    partner.save()

    treshold = partner.notify_balance_threshold or DEFAULT_NOTIFY_BALANCE_THRESHOLD
    if 0 < current_balance < treshold <= old_balance:
        return LOW_BALANCE_NOTIFICATION_MAIL_TYPE


def _fetch_current_balance(partners):
    # type: (List[Partner]) -> List[Tuple[Partner, float]]
    """
    Запрашиваем у сервиса Баланс текущие счета партнеров
    """

    log.info('start: fetch orders from balance')
    params = [
        {
            'ServiceID': BALANCE_AVIA_SERVICE_ID,
            'ServiceOrderID': p.billing_order_id
        }
        for p in partners
    ]

    orders = BalanceClient(
        env=settings.YANDEX_ENVIRONMENT_TYPE,
        client_tvm_id=settings.TVM_NAME,
        logger=log,
    ).get_orders(params)
    log.info('done: fetch orders from balance')

    def calc_curent_balance(order):
        return float(order['consume_qty']) - float(order['completion_qty'])

    billing_order_id_to_partner = {p.billing_order_id: p for p in partners}

    return [
        (billing_order_id_to_partner[order['ServiceOrderID']], calc_curent_balance(order))
        for order in orders
    ]


def _send_mail(partner, mail_type):
    # type: (Partner, Optional[str]) -> None
    if not mail_type:
        log.info('skip: send mail to partner %s', partner.code)
        return
    log.info('start: send mail for %s %s', partner.code, mail_type)
    try:
        send_mail(partner, mail_type)
    except Exception as e:
        log.exception('error: send mail for %s %s %s', partner.code, mail_type, e)
    else:
        log.info('done: send mail for %s %s', partner.code, mail_type)


def refresh_balance(partners):
    # type: (List[Partner]) -> None
    """
    Обновляем текущее состояние партнерских счетов.
    Если счет пересекает отрицательную границу, то отключаем партнера от системы.
    Если счет пересекает положительную границу, то включаем партнера в систему
       (Но только если он был отключен из-за нехватки средств)

    Также оповещаем партнеров о состоянии лицевого счета.
    """
    log.info('start')
    partners = [p for p in partners if p.billing_order_id is not None]

    result = []

    log.info('start: apply balance')
    for partner, current_balance in _fetch_current_balance(partners):
        try:
            log.info('start: apply balance for %s', partner.code)
            mail_type = _update_balance(partner, current_balance)
            result.append((partner, mail_type))
        except Exception as e:
            log.exception(u'error: apply balance for %s %s' % partner.code, e.message)
        else:
            log.info('done: apply balance for %s', partner.code)
    log.info('done: apply balance')

    log.info('start: send mails to partners')
    for partner, mail_type in result:
        _send_mail(partner, mail_type)
    log.info('done: send mails to partners')

    log.info('done')
