# coding=utf-8
from datetime import datetime
from logging import Logger, getLogger

from django.db import transaction
from retrying import retry

from travel.avia.admin.lib.yt_helpers import YtClientFabric, yt_client_fabric
from travel.avia.library.python.common.models.country_covid_info import CountryCovidInfo
from travel.avia.library.python.common.models.geo import Country


class CountryCovidInfoUpdater(object):
    """
    Обновлялка ковидных ограничений на путешествия в другие страны из-за ковида
    """
    def __init__(self, yt_client_fabric, logger):
        # type: (YtClientFabric, Logger) -> None
        self._yt_client_fabric = yt_client_fabric
        self._logger = logger

    @retry(stop_max_attempt_number=10, wait_fixed=30000)
    def update(self):
        # type: (int) -> None
        self._logger.info('Start importing covid travel restrictions')
        yt_client = self._yt_client_fabric.create()

        objects = []
        for record in yt_client.read_table('//home/rearrange/prism/travel/PRISM-68/filters'):
            try:
                country_geo_id = self.get_integer(record.get('geo_id'))
                country = Country.objects.get(_geo_id=country_geo_id)
            except:
                country = None

            if not country:
                self._logger.error('Missing geo ID: %r', record)
                continue

            country_covid_info = CountryCovidInfo(
                tourism=self.get_null_boolean(record, 'tourism'),
                quarantine=self.get_null_boolean(record, 'quarantine'),
                quarantine_days=self.get_integer(record.get('quarantine_days')),
                visa=self.get_null_boolean(record, 'visa'),
                avia=self.get_null_boolean(record, 'avia'),
                comment=record.get('text') if record.get('text') else None,
                updated_at=datetime.fromtimestamp(record.get('submitTs')/1000) if record.get('submitTs') else None,
                country=country,
            )
            objects.append(country_covid_info)
            self._logger.info('adding info: %r', country_covid_info)

        with transaction.atomic():
            CountryCovidInfo.objects.all().delete()
            CountryCovidInfo.objects.bulk_create(objects, batch_size=1000)

        self._logger.info('Done importing covid travel restrictions')

    def get_null_boolean(self, record, field_name):
        return self.get_null_boolean_internal(record.get(field_name), field_name)

    def get_null_boolean_internal(self, text_value, field_name):
        # type: (str) -> bool
        if text_value == 'no':
            return False
        if text_value == 'yes':
            return True
        if text_value is None or text_value == 'null':
            return None
        self._logger.error('Unexpected text value %s for the boolean field: %s', text_value, field_name)
        return None

    def get_integer(self, text_value):
        # type: (str) -> int
        try:
            return int(text_value)
        except:
            return 0


_logger = getLogger(__name__)

covid_country_info_updater = CountryCovidInfoUpdater(yt_client_fabric=yt_client_fabric, logger=_logger)
