# -*- coding: utf-8 -*-

import re
from importlib import import_module

from django.conf import settings
from django.core.handlers.base import BaseHandler


class MiddlewareSelector(object):
    @classmethod
    def process_request(cls, request):
        rules = cls.get_processing_rules(request)

        new_middlewares = cls.process_middleware_rules(rules)

        if 'urlconf' in rules:
            request.urlconf = rules['urlconf']

        handler = DjangoStubHandler(new_middlewares)
        handler.load_middleware()

        return handler.get_response(request)

    @classmethod
    def get_processing_rules(cls, request):
        for selector, rules in settings.MIDDLEWARE_SELECTOR:
            if cls.is_accepted_request(selector, request):
                return rules

        return {}

    @classmethod
    def is_accepted_request(cls, selector, request):
        if 'host' in selector:
            if not re.match(selector['host'], request.get_host()):
                return False

        if 'path' in selector:
            if not re.match(selector['path'], request.path):
                return False

        return True

    @classmethod
    def process_middleware_rules(cls, rules):
        new_middlewares = list(rules.get('middlewares', settings.DEFAULT_MIDDLEWARE_CLASSES))

        if 'exclude' in rules:
            new_middlewares = [
                m for m in new_middlewares if m not in rules['exclude']
            ]

        if 'filter' in rules:
            new_middlewares = [
                m for m in new_middlewares if m in rules['filter']
            ]

        return new_middlewares


class DjangoStubHandler(BaseHandler):
    """
    Заглушка для использования в MiddlewareSelector
    """
    def __init__(self, middleware_classes):
        super(DjangoStubHandler, self).__init__()

        self.middleware_classes = middleware_classes

    def load_middleware(self):
        """
        -- Populate middleware lists from settings.MIDDLEWARE_CLASSES.

        Заменили на self.middleware_classes
        """
        from django.core import exceptions
        self._view_middleware = []
        self._template_response_middleware = []
        self._response_middleware = []
        self._exception_middleware = []

        request_middleware = []
        for middleware_path in self.middleware_classes:
            try:
                mw_module, mw_classname = middleware_path.rsplit('.', 1)
            except ValueError:
                raise exceptions.ImproperlyConfigured('%s isn\'t a middleware module' % middleware_path)
            try:
                mod = import_module(mw_module)
            except ImportError as e:
                raise exceptions.ImproperlyConfigured('Error importing middleware %s: "%s"' % (mw_module, e))
            try:
                mw_class = getattr(mod, mw_classname)
            except AttributeError:
                raise exceptions.ImproperlyConfigured('Middleware module "%s" does not define a "%s" class' % (mw_module, mw_classname))
            try:
                mw_instance = mw_class()
            except exceptions.MiddlewareNotUsed:
                continue

            if hasattr(mw_instance, 'process_request'):
                request_middleware.append(mw_instance.process_request)
            if hasattr(mw_instance, 'process_view'):
                self._view_middleware.append(mw_instance.process_view)
            if hasattr(mw_instance, 'process_template_response'):
                self._template_response_middleware.insert(0, mw_instance.process_template_response)
            if hasattr(mw_instance, 'process_response'):
                self._response_middleware.insert(0, mw_instance.process_response)
            if hasattr(mw_instance, 'process_exception'):
                self._exception_middleware.insert(0, mw_instance.process_exception)

        # We only assign to this when initialization is complete as it is used
        # as a flag for initialization being complete.
        self._request_middleware = request_middleware
