# -*- coding: utf-8 -*-
import logging

import requests
import urllib3
from django.conf import settings
from requests.auth import HTTPBasicAuth

from travel.avia.library.python.common.models.geo import Settlement, Station, Station2Settlement
from travel.avia.library.python.common.models.transport import TransportType


urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)


class AbstractStatsTitleModelExporter(object):
    _model_name = None

    def __init__(self, logger):
        # type: (Logger) -> None
        self._logger = logger

    def _get_all_models(self):
        raise NotImplementedError()

    def _get_export_url(self):
        raise NotImplementedError()

    def export(self):
        # type: (str) -> None
        self._logger.info('start')

        self._logger.info('start export %s from avia database', self._model_name)
        models = self._get_all_models()
        self._logger.info('end export %s from avia database. Count objects: %s', self._model_name, len(models))

        self._logger.info('start import %s to stat database', self._model_name)

        response = requests.post(
            self._get_export_url(),
            auth=HTTPBasicAuth(settings.AVIA_STATS_IMPORT_USER, settings.AVIA_STATS_IMPORT_PASSWORD),
            json=models,
            verify=False,
        )

        self._logger.info('end import %s to stat database', self._model_name)

        response.raise_for_status()
        answer = response.json()
        self._logger.info('answer: %r', answer)

        assert answer['status'] == 'ok', 'can not import {} to stat database: {}'.format(self._model_name, answer)

        self._logger.info('end')


class AviaStatsSettlementExporter(AbstractStatsTitleModelExporter):
    _model_name = 'settlement'

    def _get_export_url(self):
        return 'https://{host}/import/settlements/'.format(host=settings.AVIA_STATS_ADMIN_HOST)

    def _get_all_models(self):
        return list(
            Settlement.objects.
            filter(id__in=self._get_settlement_ids_with_airport()).
            values('id', 'title')
        )

    def _get_settlement_ids_with_airport(self):
        airports = self._get_all_airports()

        return (
            self._get_settlement_ids_with_airport_detected_by_s2s(airports) |
            self._get_settlement_ids_with_airport_detected_by_airport_relation(airports)
        )

    def _get_all_airports(self):
        return list(
            Station.objects.filter(
                t_type_id=TransportType.PLANE_ID
            ).values('id', 'settlement_id')
        )

    def _get_settlement_ids_with_airport_detected_by_s2s(self, airports):
        return set(
            Station2Settlement.objects.
            filter(station_id__in=[a['id'] for a in airports]).
            values_list('settlement_id', flat=True)
        )

    def _get_settlement_ids_with_airport_detected_by_airport_relation(self, airports):
        return set([a['settlement_id'] for a in airports if a['settlement_id']])


class AviaStatsAirportExporter(AbstractStatsTitleModelExporter):
    _model_name = 'airport'

    def _get_all_models(self):
        return list(
            Station.objects.filter(
                t_type_id=TransportType.PLANE_ID
            ).values('id', 'title')
        )

    def _get_export_url(self):
        return 'https://{host}/import/airports/'.format(host=settings.AVIA_STATS_ADMIN_HOST)

avia_stats_settlement_exporter = AviaStatsSettlementExporter(
    logger=logging.getLogger(__name__ + '.settlement')
)

avia_stats_airport_exporter = AviaStatsAirportExporter(
    logger=logging.getLogger(__name__ + '.airport')
)
