# -*- coding: utf-8 -*-

import logging
import inspect
import os
import sys
import traceback
from copy import copy
from cStringIO import StringIO
from contextlib import contextmanager

from django.conf import settings
from django.core import mail
from django.utils import translation
from django.utils.translation import ugettext
from django.views.debug import get_exception_reporter_filter

from travel.avia.library.python.common.utils.logs import patchedHandleError  # noqa

from travel.avia.admin.lib.exceptions import SimpleUnicodeException
from travel.avia.admin.lib.fileutils import get_my_caller_file


EXPANDED_LOG_FORMAT = u"%(filename)s %(levelname)s %(asctime)s: %(message)s Line: %(lineno)d"


_created_logs = {}


class LogError(SimpleUnicodeException):
    pass


# Логи с контекстом, подключены всегда по умолчанию
class ContextRaspLogger(logging.Logger):
    """
    Специальный логгер, который позволяет дописывать в начале лога одну и туже строку.
    Строка вставляется, если используется ContextRaspFormatter
    """
    def __init__(self, *args, **kwargs):
        super(ContextRaspLogger, self).__init__(*args, **kwargs)

        import __main__ as main_module
        self.main_file_path = None

        file_path = getattr(main_module, '__file__', None)
        if not file_path:
            return

        file_path = os.path.abspath(file_path)
        if file_path.endswith('.pyc'):
            file_path = file_path[:-1]

        admin_project_path = os.path.abspath(os.path.join(__file__, '../../'))

        if file_path.startswith(admin_project_path):
            file_path = os.path.relpath(file_path, admin_project_path)

        self.main_file_path = file_path

    __context = []
    context_message = None

    def set_context(self, context=None):
        if context:
            self.__context.append(context)
        elif self.__context:
            self.__context.pop()

        self.context_message = u": ".join(self.__context)

    def add_context(self, context):
        if context:
            self.__context.append(context)
        else:
            raise LogError(u"Контекст не должен быть пустым")

        self.context_message = u": ".join(self.__context)

    def remove_context(self, context=None):
        if context:
            try:
                del self.__context[self.__context.index(context)]
            except IndexError:
                raise LogError(u"Контекст '%s' не найден в списке" % context)
        else:
            self.__context.pop()

        self.context_message = u": ".join(self.__context)

    def handle(self, record):
        if self.context_message:
            record.context = self.context_message

        record.main_file_path = self.main_file_path

        return super(ContextRaspLogger, self).handle(record)

logging.setLoggerClass(ContextRaspLogger)


class ContextRaspFormatter(logging.Formatter):
    def __init__(self, fmt=None, datefmt=None):
        if fmt:
            self._simple_fmt = fmt
        else:
            self._simple_fmt = "%(message)s"

        if ':' in self._simple_fmt:
            self._context_fmt = '%s: %%(context)s:%s' % tuple(self._simple_fmt.split(':', 1))
        else:
            self._context_fmt = self._simple_fmt + ' : ' + '%(context)s'

        self.datefmt = datefmt

    def formatException(self, ei):
        """
        Новая реализация
        """
        exc_info = traceback.format_exception(ei[0], ei[1], ei[2])
        exc_info = "".join(exc_info).strip()
        if not isinstance(exc_info, unicode):
            exc_info = str(exc_info).decode('utf8', 'replace')
        return exc_info

    def format(self, record):
        if hasattr(record, 'context') and record.context:
            self._fmt = self._context_fmt
        else:
            self._fmt = self._simple_fmt

        if record.name == '__main__' and getattr(record, 'main_file_path', None):
            record = copy(record)
            record.name = record.main_file_path

        return logging.Formatter.format(self, record)


@contextmanager
def log_context(log, context):
    log = _get_log_from_string_or_log_instance(log)

    log.add_context(context)

    try:
        yield
    finally:
        log.remove_context()


# Позволяет генерировать лог на основе места вызова функции
def get_current_file_logger_name():
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    return get_file_logger_name(caller_file)


def get_current_file_logger():
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    return logging.getLogger(get_file_logger_name(caller_file))


def get_file_logger_name(py_file):
    """
    Получаем название лога для файла проекта
    """
    if not py_file.endswith(u'.py'):
        raise LogError(u"%s не является файлом проекта не смогли создать лог" % py_file)

    file_path = os.path.abspath(py_file)
    project_path = os.path.abspath(settings.PROJECT_PATH)

    if file_path.startswith(project_path):
        relative_path = file_path.replace(project_path, u'', 1).lstrip(os.path.sep)
        logger_name = relative_path.replace(os.path.sep, u'.')
        logger_name = u'.'.join(logger_name.split(u'.')[:-1])  # remove .py(pyc) extension

        parts = logger_name.split(u'.')
        if parts[-1] == '__init__':
            del parts[-1]

            logger_name = u'.'.join(parts)

        return logger_name
    else:
        raise LogError(u"%s не является файлом проекта не смогли создать лог" % file_path)


def create_current_file_run_log(format=None, capture_stdstreams=False):
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)

    log_name = 'special.script_runs.' + get_file_logger_name(caller_file)

    return create_run_log(log_name, format, capture_stdstreams=capture_stdstreams)


def create_run_log(log_name, format=None, capture_stdstreams=False):
    from django.conf import settings
    from travel.avia.library.python.common.utils.environment import now
    format = format or settings.LOG_FORMAT

    path = os.path.join(settings.LOG_PATH, *log_name.split('.'))
    path = os.path.join(path, now().strftime('%Y-%m-%d_%H%M%S.log'))

    if not os.path.exists(os.path.dirname(path)):
        os.makedirs(os.path.dirname(path))

    handler = logging.FileHandler(path)
    handler.setFormatter(ContextRaspFormatter(format))

    log = logging.getLogger()
    log.addHandler(handler)

    if capture_stdstreams:
        capture_stdstreams_to_file(handler.stream)

    return handler.stream


def capture_stdstreams_to_file(file):
    """
    Перенаправляем стандартные потоки в этот файл
    """
    fd = file.fileno()

    sys.stdout.flush()
    os.close(1)
    os.dup2(fd, 1)

    sys.stderr.flush()
    os.close(2)
    os.dup2(fd, 2)


@contextmanager
def stdstream_file_capturing(file):
    fd = file.fileno()
    stdout_fd_bkup = os.dup(1)
    stderr_fd_bkup = os.dup(2)

    sys.stdout.flush()
    sys.stderr.flush()
    os.close(1)
    os.close(2)
    os.dup2(fd, 1)
    os.dup2(fd, 2)

    yield

    sys.stdout.flush()
    sys.stderr.flush()
    os.close(1)
    os.close(2)
    os.dup2(stdout_fd_bkup, 1)
    os.dup2(stderr_fd_bkup, 2)


capture_stdstreams = capture_stdstreams_to_file


# Функции добалений handler'ов,

def add_stdout_handler(log, format=None):
    from django.conf import settings
    format = format or settings.LOG_FORMAT
    """Добавляем handler для stdout, если еще не добавлен"""
    # Проверяем есть ли у нас handler который пишет в stdout
    add_stream_handler(log, sys.stdout, format)


def print_log_to_stdout(log=''):
    add_stdout_handler(log)


def add_stream_handler(log, stream, format=None, filters=None):
    """Добавляем handler для stream, если еще не добавлен"""

    from django.conf import settings
    format = format or settings.LOG_FORMAT

    log = _get_log_from_string_or_log_instance(log)
    filters = filters or []

    for handler in log.handlers:
        if isinstance(handler, logging.StreamHandler) and handler.stream == stream:
            return

    stream_handler = logging.StreamHandler(stream)
    stream_handler.setFormatter(ContextRaspFormatter(format))

    for log_filter in filters:
        stream_handler.addFilter(log_filter)

    log.addHandler(stream_handler)

    return stream_handler


def remove_stream_handler(log, stream):
    log = _get_log_from_string_or_log_instance(log)

    for handler in log.handlers[:]:
        if isinstance(handler, logging.StreamHandler) and handler.stream == stream:
            log.removeHandler(handler)


def add_file_handler(log, log_path, format=None, filters=None):
    """Добавляем file-handler для log_path, если еще не добавлен"""
    from django.conf import settings
    format = format or settings.LOG_FORMAT

    filters = filters or []

    for handler in log.handlers:
        if isinstance(handler, logging.FileHandler) and handler.baseFilename == log_path:
            return

    try:
        handler = logging.FileHandler(log_path, 'a', encoding='utf-8')
        handler.setFormatter(ContextRaspFormatter(format))

        for log_filter in filters:
            handler.addFilter(log_filter)

        log.addHandler(handler)
    except IOError:
        add_stdout_handler(log, format)


def get_rasp_log_path(log_name):
    name = log_name.strip(u'.')
    filename = name.split('.')[-1]
    dirs = name.split('.')[:-1]

    logpath = settings.LOG_PATH
    logpath = os.path.join(logpath, *dirs)
    if not os.path.exists(logpath):
        os.makedirs(logpath)

    log_filename = os.path.join(logpath, filename + '.log')
    log_path = os.path.abspath(log_filename)

    return log_path


class CollectorHandler(logging.StreamHandler):
    def __init__(self):
        stream = StringIO()
        super(CollectorHandler, self).__init__(stream)

    def get_collected(self, clean=True):
        data = self.stream.getvalue().decode('utf8')

        if clean:
            self.stream = StringIO()

        return data


def get_collector(log_or_logname, format=None, log_level=logging.INFO, lang='ru'):
    from django.conf import settings
    format = format or settings.LOG_COLLECTOR_FORMAT

    log = _get_log_from_string_or_log_instance(log_or_logname)

    if lang == 'ru':
        handler = CollectorHandler()

    elif lang == 'en':
        handler = EnCollectorHandler()

    else:
        raise NotImplementedError('Unknown language for log collector')

    handler.setLevel(log_level)
    handler.setFormatter(ContextRaspFormatter(format))
    log.addHandler(handler)

    return handler


def remove_collector(log_or_logname, collector):
    log = _get_log_from_string_or_log_instance(log_or_logname)
    log.removeHandler(collector)


class LangCollectors(object):
    def __init__(self, log_or_logname, format=None, log_level=logging.INFO, languages=None):
        from django.conf import settings

        self.log_or_logname = log_or_logname
        self.format = format or settings.LOG_FORMAT
        self.log_level = log_level
        self.languages = languages or ('ru',)

        self.handlers = dict()

    def __enter__(self):
        for lang in self.languages:
            self.handlers[lang] = get_collector(self.log_or_logname, self.format,
                                                self.log_level, lang)
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        for lang in self.languages:
            remove_collector(self.log_or_logname, self.handlers[lang])

    def get_collected(self, lang, clean=True):
        return self.handlers[lang].get_collected(clean)


# Для записи логов на разных языках

class EnMixin(object):
    def format(self, record):
        with translation.override('en'):
            copy_of_record = copy(record)

            if isinstance(copy_of_record.msg, basestring):
                copy_of_record.msg = ugettext(copy_of_record.msg)

            return super(EnMixin, self).format(copy_of_record)


class EnFileHandler(EnMixin, logging.FileHandler):
    pass


class EnCollectorHandler(EnMixin, CollectorHandler):
    pass


@contextmanager
def get_collector_context(log_or_logname='', format=None, level=None):
    log_collector = get_collector(log_or_logname, format=format)

    level = level if level is not None else logging.INFO
    log_collector.setLevel(level or logging.INFO)
    try:
        yield log_collector

    finally:
        remove_collector(log_or_logname, log_collector)


# Копипаст handler'а рассылающего письма

class AdminEmailHandler(logging.Handler):
    """An exception log handler that emails log entries to site admins.

    If the request is passed as the first argument to the log record,
    request data will be provided in the
    """

    def emit(self, record):
        if '\n' in record.msg:
            message_subject, message_data = record.msg.split('\n', 1)

            if message_subject.endswith(':'):
                message_subject = message_subject[:-1]
        else:
            message_subject, message_data = record.msg, None

        try:
            request = record.request
            subject = '%s: %s' % (
                record.levelname,
                message_subject
            )
            filter = get_exception_reporter_filter(request)
            request_repr = filter.get_request_repr(request)
        except:
            subject = '%s: %s' % (
                record.levelname,
                message_subject
            )

            request = None
            request_repr = "Request repr() unavailable"

        if record.exc_info:
            stack_trace = '\n'.join(traceback.format_exception(*record.exc_info))
        else:
            stack_trace = 'No stack trace available'

        message = "%s\n\n%s" % (stack_trace, request_repr)

        boilerplate = []

        if request:
            try:
                boilerplate.append(request.build_absolute_uri())
            except:
                boilerplate.append("request.build_absolute_uri() failed")

        if record.exc_info:
            type_, value, _ = record.exc_info

            if value:
                try:
                    boilerplate.append("%s: %s" % (type_.__name__, value))
                except:
                    boilerplate.append("exception formatting failed")

        boilerplate.append(settings.PKG_VERSION)

        message = "%s\n\n%s" % ("\n".join(boilerplate), message)

        if message_data:
            message = "%s\n\n%s" % (message_data, message)

        mail.send_mail(subject, message, settings.SERVER_EMAIL, settings.ERRORS_RECIPIENTS, fail_silently=True)


# Вспомогательные функции
def _get_log_from_string_or_log_instance(log_or_logname):
    if isinstance(log_or_logname, basestring) or log_or_logname is None:
        return logging.getLogger(log_or_logname)
    else:
        return log_or_logname


def remove_secrets(params_dict):
    """
    Скрываем чувствительные данные в словаре

    :param dict[str, str] params_dict: словарь с данными
    :return:
    """
    cp = params_dict.copy()
    secret_keys = ['password', 'passwd', 'hash']
    for key in secret_keys:
        if key in cp:
            cp[key] = '*' * len(cp[key])
    return cp
