# -*- coding: utf-8 -*-
import calendar
import threading
from abc import abstractmethod, ABCMeta
from datetime import datetime, timedelta
from itertools import izip

import os
import requests
import requests.auth
import yt.wrapper as yt
from django.conf import settings

from travel.avia.admin.avia_scripts.fetch_query_rules import SettlementCache
from travel.avia.admin.lib.logs import remove_secrets

# Загрузить кеш маппингов iata -> город
settlement_cache = SettlementCache.precached()


class MarkerWriter(object):
    def __init__(self, source, logger):
        # Мютекс сделан на будушее для thread-safe
        self._mutex = threading.Lock()
        self._yt_root = settings.YT_PARTNER_BOOKING_ROOT
        self._rows = []
        self._logger = logger
        self._unixtime = calendar.timegm(datetime.utcnow().timetuple())
        self.source = source

    def add_rows(self, rows):
        with self._mutex:
            self._rows.extend(rows)

    @property
    def _json_rows(self):
        for row in self._rows:
            yt_row = {
                'unixtime': self._unixtime,
                'partner': row.partner,
                'marker': row.marker,
                'status': row.status,
                'created_at': self._json_dt(row.created_at),
                'confirm_dt': self._json_dt(row.confirm_dt),
                'source': self.source,
                'partner_id': row.partner_id,
                'billing_order_id': row.billing_order_id,
                'order_id': row.order_id,
                'price': row.price,
                'currency': row.currency,
                'from': row.airport_from,
                'to': row.airport_to,
                'trip_type': row.trip_type,
                'ticket_number': row.ticket_number,
            }

            yield yt_row

    def _json_dt(self, dt):
        return dt.strftime('%Y-%m-%d %H:%M:%S') if dt else None

    def write_to_yt(self, report_date):
        yt_table = yt.TablePath(
            os.path.join(self._yt_root, report_date.strftime('%Y-%m-%d')),
            append=True,
        )

        if not self._rows:
            self._logger.info('Nothing to write to: %s', yt_table)
            return

        if not yt.exists(self._yt_root):
            self._logger.info('Creating cypress for %s', self._yt_root)
            yt.create('map_node', self._yt_root)

        if not yt.exists(yt_table):
            self._logger.info('Creating table %s', yt_table)
            yt.create_table(
                yt_table,
                attributes={
                    'schema': [
                        {
                            'name': 'unixtime',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'source',
                            'type': 'string',
                        },
                        {
                            'name': 'partner',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'partner_id',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'billing_order_id',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'marker',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'status',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'created_at',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'confirm_dt',
                            'type': 'string',
                        },
                        {
                            'name': 'order_id',
                            'type': 'string',
                        },
                        {
                            'name': 'price',
                            'type': 'double',
                        },
                        {
                            'name': 'currency',
                            'type': 'string',
                        },
                        {
                            'name': 'from',
                            'type': 'string',
                        },
                        {
                            'name': 'to',
                            'type': 'string',
                        },
                        {
                            'name': 'trip_type',
                            'type': 'string',
                        },
                        {
                            'name': 'ticket_number',
                            'type': 'string',
                        }
                    ],
                }
            )

        self._flush_rows(yt_table)

    def _create_key(self, row):
        ticket_number = row.get('ticket_number')
        return (
            row['created_at'],
            row['marker'],
            row['status'],
            row['source'],
            row['partner'],
            ticket_number if ticket_number else 'unknown',
        )

    def _filter_duplicated_rows(self, yt_table, rows):
        if yt.exists(yt_table):
            uniq_keys = {self._create_key(r) for r in yt.read_table(yt_table, format=yt.format.JsonFormat())}
        else:
            uniq_keys = set()

        self._logger.info('Start: filter duplicate rows. New rows: %d', len(rows))

        temp = []
        for row in rows:
            key = self._create_key(row)
            if key not in uniq_keys:
                temp.append(row)
                uniq_keys.add(key)

        self._logger.info('Finish: filter duplicate rows. Uniq new rows: %d', len(temp))

        return temp

    def _filter_broken_rows(self, rows):
        self._logger.info('Start: filter broken rows. Uniq new rows: %d', len(rows))
        temp = []
        for r in rows:
            if not all(field for field in self._create_key(r)):
                self._logger.info('Filter row %r, because some required field is empty', r)
                continue
            temp.append(r)

        self._logger.info('Finish: filter broken rows. Uniq new rows: %d', len(temp))

        return temp

    def _flush_rows(self, yt_table):
        self._logger.info('Write YT table: %s', yt_table)

        with self._mutex:
            rows = list(self._json_rows)
            rows = self._filter_duplicated_rows(yt_table, rows)
            rows = self._filter_broken_rows(rows)

            yt.write_table(yt_table, rows, format=yt.YsonFormat())
            self._rows = []

    class Row(object):
        def __init__(self, partner, marker, status, created_at, confirm_dt=None, partner_id=None,
                     billing_order_id=None, order_id=None, price=None, currency=None, airport_from=None,
                     airport_to=None, trip_type=None, ticket_number=None):
            self._status_choices = {'booking', 'paid', 'cancel'}
            self._trip_type_choices = {'oneway', 'roundtrip', 'openjaw', None}

            if not isinstance(created_at, datetime):
                raise TypeError('created_at=%r' % created_at)

            if confirm_dt and not isinstance(confirm_dt, datetime):
                raise TypeError('confirm_dt=%r' % confirm_dt)

            if status not in self._status_choices:
                raise self.BadStatusChoise(repr(status))

            if trip_type not in self._trip_type_choices:
                raise self.BadTripTypeChoise(repr(trip_type))

            if not marker:
                raise self.EmptyMarker()

            self.created_at = created_at
            self.confirm_dt = confirm_dt
            self.marker = marker
            self.status = status
            self.partner = partner
            self.partner_id = int(partner_id)
            self.billing_order_id = int(billing_order_id)
            self.order_id = order_id
            self.price = float(price) if price is not None else None
            self.currency = currency
            self.airport_from = airport_from
            self.airport_to = airport_to
            self.trip_type = trip_type
            self.ticket_number = ticket_number

        class BadStatusChoise(Exception):
            pass

        class BadTripTypeChoise(Exception):
            pass

        class EmptyMarker(Exception):
            pass

        def __repr__(self):
            return 'Row object [%s]' % vars(self)


class MarkerTransfer(object):
    def __init__(self, partner, marker_writer, marker_reader, logger):
        """
        :type partner: travel.avia.library.python.common.models.partner.Partner
        :type marker_writer: lib.marker.MarkerWriter
        :type marker_reader: lib.marker.BaseMarkerReader
        :type logger: logging.Logger
        """
        self._partner = partner
        self._marker_writer = marker_writer
        self._marker_reader = marker_reader
        self._logger = logger

    def transfer(self, date):
        """
        Принимаем данные из marker_reader и передаем в marker_writer
        :type date: datetime
        :param date: дата, на которую нужно брать данные у партнера и перекладывать в YT
        """
        try:
            orders = self._marker_reader.import_data(date)
            self.export_data(orders, date)
        except Exception:
            self._logger.exception(
                'Error transfering data for partner %s, date %s',
                self._partner,
                date,
            )
        else:
            self._logger.info('Data transfer complete')

    def export_data(self, orders, date):
        """
        :type orders: list[dict[str,Any]]
        :param orders: список заказов. Формат смотри в BaseMarkerReader.parse_report
        :type date: datetime
        :param date: дата на которую нужно записать данные в YT
        """
        rows = []
        for order in orders:
            try:
                row = MarkerWriter.Row(
                    partner=self._partner.code,
                    partner_id=self._partner.id,
                    billing_order_id=self._partner.billing_order_id,
                    order_id=order.get('order_id'),
                    created_at=order.get('created_at'),
                    price=order['price'],
                    currency=order.get('currency'),
                    airport_from=order.get('airport_from'),
                    airport_to=order.get('airport_to'),
                    status=order['status'],
                    marker=order['marker'],
                    confirm_dt=order.get('confirm_date'),
                    trip_type=order.get('trip_type'),
                    ticket_number=order.get('ticket_number'),
                    # confirm_dt - кандидат на удаление
                    # (смотри: https://st.yandex-team.ru/RASPTICKETS-13086#1536754443000)
                )
                rows.append(row)
            except Exception:
                self._logger.exception('Problems with order %s', order)

        self._marker_writer.add_rows(rows)
        self._marker_writer.write_to_yt(date)


class BaseMarkerReader(object):
    __metaclass__ = ABCMeta

    def __init__(self, statuses_map, logger):
        """
        :type logger: logging.Logger
        :param logger: loggin.Logger логгер для проброса сообщений и ошибок
        :type statuses_map: dict[str, str]
        :param statuses_map: словарь преобразования статусов от партнера в наши
        """
        self._logger = logger
        self._statuses_map = statuses_map

    @abstractmethod
    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :param datetime.date date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """

    @abstractmethod
    def parse_report(self, content):
        """
        Каждый подкласс должен переопределять этот метод и возвращать
        массив заказов со следующей структурой
        [{
            order_id: ...,

            created_at: ...,  --datetime format

            confirm_date: ...,  --datetime format

            price: ...,

            currency: ...,

            status: ...,  (booking, paid, cancel)

            marker: ...,

            airport_from: ...,  *** IATA

            airport_to: ...,  *** IATA

            trip_type: ..., 'oneway' | 'roundtrip' | 'openjaw' ***

        }, ...]

        Поля From, To и TripType (помеченные звездочкой) могут быть автоматически
        заполнены. Для автоматического заполнения нужно добавить по ключу flights список аэопортов,
        через которые проходил маршрут, а также даты перелетов и вызвать fillin_airports.
        Подробнее структура, которую нужно заполнить описана в get_airports.
        Вызывать fillin_airport каждый наследник должен сам внутри parse_report

        :type content: bytes
        :param content: сырые данные от поставщика
        :rtype: list[dict[str, Any]]
        """

    def _parse_status(self, status):
        """
        Преобразуем строчку статуса, которую нам передает партнер в наш статус
        При создании экземпляра класса нужно внимательно следить за statuses_map.
        На данном этапе соответствие статусам в YT не проверяется, поэтому
        оно может упасть позже, когда этот статус будет передан MarkerWriter

        :type status: str
        :param status: Статус от партнера
        :rtype: str
        :return: наш статус для YT (см lib.marker.MarkerWriter.Row)
        """
        parsed = self._statuses_map.get(status)
        if status and not parsed:
            raise Exception('Unknown status: {}'.format(status))
        return parsed

    def _fillin_airports(self, orders):
        """
        Принимаем словарик заказов и заполняем начальную и конечную точку маршрута.
        Информацию о структуре входных данных смотрим в _get_airports

        :type orders: list[dict[str, Any]]
        :param orders: список заказов в виде словарей.
        """
        for order in orders:
            (From, To), trip_type = self._get_airports(order)
            order['airport_from'] = From
            order['airport_to'] = To
            order['trip_type'] = trip_type
        return None

    def _get_airports(self, order):
        """
        Получаем из неупорядоченного списка сегментов пункт начала маршрута и пункт конца.
        Проверяем, что это маршрут закольцованный и если так, то ищем середину
        Проверяем, что это один цельный маршрут.

        Формат структуры по ключу flights:
        [{
            from: ..., -- IATA код аэропорта отлета

            to: ...,  -- IATA код аэропорта прилета

            departure_dt: ...,  --дата и время отлета datetime

            arrival_dt: ...,  --дата и время прилета datetime
        }]

        :type order: dict[str, list[dict[str, Any]]]
        :param order: Заказ, в котором по ключу flights лежит список полетов
        :rtype: ((str, str), str)
        :return: ((начальная точка (IATA), конечная точка (IATA)), тип поездки)
        """
        flights = sorted(order['flights'], key=lambda flight: flight['departure_dt'])

        if not flights:
            raise Exception('No segments found')

        orig = flights[0]['from']
        dest = flights[-1]['to']

        trip_type = None
        try:
            if self._is_round_trip(flights):
                dest = self._most_delayed_airport(flights)
                trip_type = 'roundtrip'
            else:
                trip_type = 'oneway'

            if not self._is_continuous(flights):
                trip_type = 'openjaw'
        except:
            self._logger.exception('Could not determine trip type')

        return (orig, dest), trip_type

    def _is_round_trip(self, flights):
        """
        Определяем, прилетели ли мы в тот же город, из которого улетали

        :type flights: list[dict[str,Any]]
        :param flights: Список словарей с ключами from и to и значениями в виде IATA кодов
        :rtype: bool
        :return: True если вернулись откуда прилетели, иначе False
        """
        if not flights or len(flights) == 1:
            return False
        try:
            orig = flights[0]['from']
            dest = flights[-1]['to']
            if orig is None or dest is None:
                return False
            if orig == dest or self._city(orig) == self._city(dest):
                return True
            return False
        except Exception:
            self._logger.warn('Problem detecting round trip in fligh %s', str(flights))
            raise

    def _is_continuous(self, flights):
        """
        Определяем, является ли маршрут географически непрерывным,
        то есть нет прилета в один город, а вылета из другого

        :type flights: list[dict[str, Any]]
        :param flights: список словарей полетов по маршруту с ключами from и to.
        Значения ключей from и to - IATA коды аэропорта
        :return: True если полет нерперывный, иначе False
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            return True
        for (past, future) in izip(flights, flights[1:]):
            inport = past['to']
            outport = future['from']
            if inport is None or outport is None:
                self._logger.info('Unknown airport in %s', flights)
                return False

            if inport != outport and self._city(inport) != self._city(outport):
                return False
        return True

    def _most_delayed_airport(self, flights):
        """
        Находим среди всех перелетов аэропорт, в котором находились дольше всего
        Необходимо при поиске направления в случае round trip

        :type flights: list[dict[str, Any]]
        :param flights: список словарей с ключами from (IATA), to (IATA), deprature_dt (datetime),
        arrival_dt (datetime)
        :rtype: str
        :return: Аэропорт, в котором была наибольшая задержка между перелетами
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            raise Exception('Cannot determine for a single flight')

        max_delay = timedelta(0)
        max_delay_airport = None
        for (past, future) in izip(flights, flights[1:]):
            delay = future['departure_dt'] - past['arrival_dt']
            if delay > max_delay:
                max_delay = delay
                if future['from'] is not None:
                    max_delay_airport = future['from']
                else:
                    max_delay_airport = past['to']
        return max_delay_airport

    def _city(self, iata):
        try:
            return settlement_cache.settlement(iata)
        except Exception:
            self._logger.exception('Problems with airport %s', str(iata))
            raise


class MarkerHTTPReader(BaseMarkerReader):
    __metaclass__ = ABCMeta

    def __init__(self, statuses_map, logger):
        """
        :type logger: logging.Logger
        :param logger: loggin.Logger логгер для проброса сообщений и ошибок
        :type statuses_map: dict[str, str]
        :param statuses_map: словарь преобразования статусов от партнера в наши
        """
        super(MarkerHTTPReader, self).__init__(
            logger=logger,
            statuses_map=statuses_map,
        )

    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :type date: datetime
        :param date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        (url, params) = self._get_request_data(date)
        self._logger.info('Get %s, %s', url, remove_secrets(params))
        r = requests.get(
            url,
            auth=self._get_request_auth(),
            params=params,
        )
        return self.parse_report(r.content)

    @abstractmethod
    def _get_request_auth(self):
        """
        :rtype: requests.auth.AuthBase
        :return: тип аутентификации у партнера, например
        requests.auth.HTTPBasicAuth
        """

    @abstractmethod
    def _get_request_data(self, report_date):
        """
        Возвращает кортеж идентифицирующий место, откуда брать данные для
        текущей даты

        :type report_date: datetime
        :param report_date: дата на которую нужены данные
        :rtype: (str, dict[str, str])
        :return: (url, params) \
        url - адрес, по которому нужно запрашивать данные \
        params - словарь параметров, которые нужно передавать по url
        """


def get_secret(partner, key):
    """
    Забираем секреты из окружения. Падаем, если их нет

    :type partner: str
    :param partner: имя партнера
    :type key: str
    :param key: секрет партнера
    :rtype: str
    :return: секрет из окружения
    """
    env_secrets = 'AVIA_ADMIN_SECRETS'
    secret_key = '_'.join([env_secrets, partner.upper(), key.upper()])
    value = os.environ.get(secret_key)
    if not value:
        raise Exception('Please, put a secret %s in environment' % secret_key)
    return value
