# -*- coding: utf-8 -*-
import logging
from urllib2 import URLError

from django.conf import settings
from django.contrib.auth.models import User, AnonymousUser
from django.http import HttpResponseRedirect
from django.utils.http import urlencode

from blackbox import XmlBlackbox, BlackboxError, FIELD_LOGIN

from travel.library.python.tvm_ticket_provider import provider_fabric


logger = logging.getLogger(__name__)

# http://wiki.yandex-team.ru/passport/MDAintro#chtotakoemda

FIELD_LANG = ('userinfo.lang.uid', 'lang')


blackbox = XmlBlackbox(
    url=settings.YANDEX_TEAM_BLACKBOX_URL,
    dbfields=[FIELD_LOGIN, FIELD_LANG],
)


client = provider_fabric.create(settings)


class YandexUser(object):
    """
    Модель яндексового пользователя.
    """
    def __init__(self, uid, is_lite, fields, need_reset):
        self.uid = uid
        self.is_lite_uid = is_lite
        self.fields = fields
        self.need_reset = need_reset

    def is_authenticated(self):
        return not self.is_lite()

    def is_lite(self):
        return bool(self.is_lite_uid)

    def __getattr__(self, field):
        try:
            return self.fields[field]
        except KeyError:
            raise AttributeError(field)

    def get_django_user(self):
        try:
            return User.objects.get(username=self.login)
        except User.DoesNotExist:
            return AnonymousUser()

    def __str__(self):
        return self.login


class AnonymousYandexUser(object):
    """
    Заглушка незалогиненного пользователя.
    """
    login = '(Not logged in)'
    uid = None
    lang = None

    def is_authenticated(self):
        return False

    def is_lite(self):
        return False

    def __str__(self):
        return self.login

    def get_django_user(self):
        return AnonymousUser()


def add_yauser(request):
    """Возвращаемое значение: нужен-ли редирект на MDA"""

    request.yauser = AnonymousYandexUser()

    if settings.DISABLE_YAUTH:
        return False

    sessionid = request.COOKIES.get('Session_id')
    userip = request.META.get('REMOTE_ADDR', None) or request.META.get('HTTP_X_REAL_IP', None)
    host = request.META.get('HTTP_HOST')

    headers = None
    ticket = client.get_ticket(settings.BLACKBOX_TVM_NAME)
    if ticket:
        headers = {'X-Ya-Service-Ticket': ticket}

    if sessionid is not None:
        try:
            session = blackbox.sessionid(sessionid, userip, host, headers=headers)

            # Кука Session_id есть и имеет статус VALID, то пользователь авторизован
            if session.valid:
                if session.uid is not None:
                    request.yauser = YandexUser(int(session.uid), False, session.fields, session.redirect)

        except (URLError, BlackboxError):  # если blackbox недоступен, то пользователь анонимен
            logger.exception(u'Ошибка авторизации')


class MDAMiddleware:
    def process_request(self, request):
        add_yauser(request)

        if settings.DISABLE_YAUTH:
            return

        if request.method != 'GET':
            return

        if request.is_ajax():
            return

        if request.yauser.get_django_user().is_authenticated():
            return

        # FIXME: пока убираем редирект, т.к. для некоторых ручек не нужна обязательная авторизация
        return

        redirect_url = 'https://passport.yandex-team.ru/passport?{}'.format(urlencode(dict(
            mode='auth', retpath=request.build_absolute_uri()
        )))

        return HttpResponseRedirect(redirect_url)
