import functools
import gzip
import itertools
from collections import defaultdict
from cStringIO import StringIO
from urlparse import urljoin

import requests

from travel.avia.admin.lib.network import get_ipv6_addresses_by_hostname


class CollectingError(Exception):
    def __init__(self, message='', partner=None, host=None, exception=None):
        super(CollectingError, self).__init__(message)
        self.partner = partner
        self.host = host
        self.exception = exception

    def __str__(self):
        return 'CollectingError<msg={}, partner={}, host={}, exception={}>'.format(
            self.args[0], self.partner, self.host, self.exception
        )

    @property
    def message(self):
        return self.args[0]


class PartnerErrorInfo(object):
    __slots__ = ['error_type', 'request_response', 'response_content', 'error']

    def __init__(self, error_type, request_response, response_content, error):
        self.error_type = error_type
        self.request_response = request_response
        self.response_content = response_content
        self.error = error


class InstancePartnerErrorsCollector(object):
    def __init__(self, host, logger):
        """
        Class for collecting partners errors from one instanse
        :param host: address or hostname of the instance
        :param logger: logging.Logger InstancePartnerErrorsCollector
        """

        self._base_url = 'http://{host}/api/1.0/track/bad/'.format(host=host)
        self._host = host
        self._logger = logger
        self._session = None

    @property
    def session(self):
        if self._session is None:
            self._session = requests.Session()

        return self._session

    def _request(self, url):
        r = self.session.get(urljoin(self._base_url, url))
        r.raise_for_status()

        return r

    def _json_request(self, url):
        r = self._request(url).json()
        if r['status'] != 'ok':
            raise CollectingError('Status {} for url {}'.format(r.status_code, url), host=self._host)

        return r['data']

    def get_available_partners(self):
        return set(self._json_request(''))

    def collect(self, partners):
        available_partners = self.get_available_partners()
        self._logger.info('Available partners: %s', ', '.join(available_partners))
        partners_to_ask = set(partners) & available_partners
        return {
            partner: self._collect_for_partner(partner)
            for partner in partners_to_ask
        }

    def _collect_for_partner(self, partner):
        for exception in self._get_partner_exceptions(partner):
            if exception.endswith('.lock'):
                continue
            for error in self._get_partner_errors(partner, exception):
                yield error

    def _get_partner_errors(self, partner, exception):
        base_url = partner + '/' + exception + '/'
        try:
            all_files = set(self._json_request(base_url))
        except Exception:
            self._logger.exception('Could not parse errors on url %s', base_url)
            return

        self._logger.info('Found files for %s: %s', partner, ', '.join(all_files))
        processed_indices = set()
        for file_name in all_files:
            try:
                index = int(file_name.split('_', 1)[0])
            except ValueError:
                self._logger.warning('Bad filename {} for partner {} exception {} on host {}'.format(
                    file_name, partner, exception, self._host,
                ))
                continue

            if index in processed_indices:
                continue
            processed_indices.add(index)

            request_response = '{}_request_response.txt.gz'.format(index)
            response_content = '{}_response_content.txt.gz'.format(index)
            error = '{}_error.txt.gz'.format(index)

            def download_if_exists(filename):
                return self._gzip_request(base_url + filename) if filename in all_files else None

            yield PartnerErrorInfo(
                error_type=exception,
                request_response=download_if_exists(request_response),
                response_content=download_if_exists(response_content),
                error=download_if_exists(error),
            )

    def _gzip_request(self, url):
        r = self._request(url)
        return gzip.GzipFile(fileobj=StringIO(r.content)).read()

    def _get_partner_exceptions(self, partner):
        return self._json_request(partner + '/')


class PartnerErrorsCollector(object):
    def __init__(self, td_main_host, logger):
        """
        Class for collecting partners errors from ticket-daemon hosts
        :param td_main_host: Qloud service-discovery component hostname, e.g. gunicorn.testing.avia-ticket-daemon.avia.stable.qloud-d.yandex.net
        :param logger: logging.Logger instance
        """

        self._td_main_host = td_main_host
        self._logger = logger

    def collect(self, partners, max_per_partner=12, max_per_type=3):
        """
        Collect errors form instances
        :param partners: iterable of partner codes
        :param max_per_partner: maximum number of errors for each partner
        :param max_per_type: maximum number of errors for each error type
        """
        partner_errors = defaultdict(functools.partial(ErrorStore, max_per_partner, max_per_type))
        partners_to_ask = set(partners)
        for instance in self._ticket_daemon_instanses():
            self._logger.info('Collecting from instance %s', instance)
            collector = InstancePartnerErrorsCollector(instance, self._logger)
            try:
                documents = collector.collect(partners_to_ask)
            except Exception, exc:
                self._logger.error('Exception %r on collecting from instance %s', exc, instance)
                continue
            for partner, errors in documents.iteritems():
                error_store = partner_errors[partner]
                if error_store.is_full():
                    partners_to_ask.remove(partner)
                    continue

                for error in errors:
                    error_store.put_error(error.error_type, error)
                    if error_store.is_full():
                        partners_to_ask.remove(partner)
                        break

        return {
            partner: store.get_errors()
            for partner, store in partner_errors.iteritems()
        }

    def _ticket_daemon_instanses(self):
        for host in get_ipv6_addresses_by_hostname(self._td_main_host):
            yield '[' + host + ']'

    @staticmethod
    def _validate_partner_error(error):
        return bool(error.request_response)


class ErrorStore(object):
    def __init__(self, max_errors, max_errors_per_type):
        self.max_errors = max_errors
        self.max_errors_per_type = max_errors_per_type
        self.errors = defaultdict(list)
        self._error_count = 0

    def is_full(self):
        return self._error_count >= self.max_errors

    def put_error(self, error_type, error):
        errors = self.errors[error_type]
        if len(errors) >= self.max_errors_per_type:
            return False

        errors.append(error)
        self._error_count += 1

    def get_errors(self):
        return list(itertools.chain(*self.errors.itervalues()))
