# coding=utf-8
from logging import Logger, getLogger
from collections import defaultdict

from django.db import transaction
from django.conf import settings

from travel.avia.admin.lib.redirect_provider import (
    RedirectProvider,
    redirect_provider
)
from travel.avia.admin.lib.yt_helpers import YtTableProvider, yt_table_provider

if settings.configured:
    from travel.avia.library.python.avia_data.models import (
        NationalVersion, PopularPartners, PopularPartnersByRoute
    )


class PopularPartnersBuilder(object):
    """
    Обновлялка списка популярных партнеров
    1) по нац версии
    2) по направлению в каждой нац версии
    """
    def __init__(self, redirect_provider, yt_table_provider, logger):
        # type: (RedirectProvider, YtTableProvider, Logger) -> None
        self._redirect_provider = redirect_provider
        self._yt_table_provider = yt_table_provider
        self._logger = logger

    def update(self, precise, route_precise, partner_precise):
        # type: (int, float, float) -> None
        """
        :param precise: За сколько дней забираем данные, для расчета таблиц
        :param route_precise: Сколько направлений считать значимыми в процентах
        :param partner_precise: Сколько партеров считать значимыми в процентах
        :return: None
        """
        self._logger.info('Start')

        self._logger.info('start fetch source table')
        sources = self._yt_table_provider.get_last(
            path='//home/avia/logs/avia-redir-log',
            days=precise
        )
        self._logger.info('input sources: %r', sources)
        self._logger.info('finish fetch source table')

        self._logger.info('start fetch data')
        nv_to_clicks_by_route = self._redirect_provider.collect(
            sources
        )

        self._logger.info('finish fetch data')

        self._logger.info('start build new models')
        popular_partners, route_popular_partners = self._build_new_models(
            nv_to_clicks_by_route,
            route_precise,
            partner_precise
        )
        self._logger.info('finish build new models')

        self._logger.info('start update db')
        with transaction.atomic():
            PopularPartners.objects.all().delete()
            PopularPartners.objects.bulk_create(
                popular_partners
            )

            PopularPartnersByRoute.objects.all().delete()
            PopularPartnersByRoute.objects.bulk_create(
                route_popular_partners
            )

        self._logger.info('finish update db')

    def _build_new_models(self, nv_to_clicks_by_route,
                          route_precise, partner_precise):

        popular_partners_list = []
        route_popular_partners_list = []

        for nv, nv_id in NationalVersion.objects.values_list('code', 'id'):
            clicks_by_route = nv_to_clicks_by_route.get(nv)
            if not clicks_by_route:
                continue

            popular_partners = list(
                self._calculate_partner_popular(
                    route_to_clicks_by_partner=clicks_by_route,
                    nv_id=nv_id,
                    partner_precise=partner_precise
                )
            )

            route_popular_partners = list(
                self._calculate_route_partner_popular(
                    route_to_clicks_by_partner=clicks_by_route,
                    nv_id=nv_id,
                    route_precise=route_precise,
                    partner_precise=partner_precise
                )
            )

            self._logger.info('nv: [%s], popular_partners: [%d], '
                              'route_popular_partners: [%d]',
                              nv, len(popular_partners),
                              len(route_popular_partners))

            popular_partners_list += popular_partners
            route_popular_partners_list += route_popular_partners

        self._logger.info(
            'popular_partners: [%d], '
            'route_popular_partners: [%d]',
            len(popular_partners_list), len(route_popular_partners_list)
        )

        return popular_partners_list, route_popular_partners_list

    def _calculate_partner_popular(self,
                                   route_to_clicks_by_partner,
                                   nv_id,
                                   partner_precise):
        partner_to_clicks = defaultdict(int)
        for clicks_by_partner in route_to_clicks_by_partner.itervalues():
            for partner_id in clicks_by_partner:
                partner_to_clicks[partner_id] += clicks_by_partner[partner_id]
        partner_to_clicks = dict(partner_to_clicks)

        for partner_id, score in self._reduce_important(partner_to_clicks,
                                                        partner_precise):
            yield PopularPartners(
                national_version_id=nv_id,
                partner_id=int(partner_id),
                score=score
            )

    def _calculate_route_partner_popular(self,
                                         route_to_clicks_by_partner,
                                         nv_id,
                                         route_precise,
                                         partner_precise):
        route_to_clicks = {
            route: sum(clicks.itervalues(), 0)
            for route, clicks in route_to_clicks_by_partner.iteritems()
        }
        top_route_and_score = self._reduce_important(
            route_to_clicks,
            precise=route_precise
        )

        for route, _ in top_route_and_score:
            for partner_id, score in self._reduce_important(
                    route_to_clicks_by_partner[route],
                    partner_precise
            ):
                from_type, from_id = self._parse_point_key(route[0])
                to_type, to_id = self._parse_point_key(route[1])

                yield PopularPartnersByRoute(
                    from_type=from_type,
                    from_id=from_id,
                    to_type=from_type,
                    to_id=to_id,
                    national_version_id=nv_id,
                    partner_id=int(partner_id),
                    score=score
                )

    def _parse_point_key(self, point_key):
        try:
            point_type = None

            if point_key[0] == 's':
                point_type = PopularPartnersByRoute.STATION_TYPE
            if point_key[0] == 'c':
                point_type = PopularPartnersByRoute.SETTLEMENT_TYPE

            if point_type is None:
                return None
            return point_type, int(point_key[1:])
        except Exception:
            return None

    def _reduce_important(self, key_to_clicks, precise):
        clicks = 0
        total_clicks = sum(key_to_clicks.itervalues(), 0)
        max_clicks = total_clicks * precise

        routes = sorted(
            key_to_clicks.keys(),
            key=lambda r: key_to_clicks[r], reverse=True
        )

        result = []
        for route in routes:
            if clicks > max_clicks:
                break
            clicks += key_to_clicks[route]
            result.append((route, float(key_to_clicks[route]) / total_clicks))

        return result


popular_partners_builder = PopularPartnersBuilder(
    redirect_provider=redirect_provider,
    yt_table_provider=yt_table_provider,
    logger=getLogger(__name__)
)
