# -*- coding: utf-8 -*-
import logging
import os

import boto3
import fnmatch
import magic
from django.conf import settings

log = logging.getLogger(__name__)


class S3SyncError(Exception):
    pass


def bucket_exists(s3, buck):
    buckets = [bucket for bucket in s3.buckets.all()]
    return True if buck in buckets else False


def s3sync(config):
    endpoint_url = settings.MDS_S3_ENDPOINT_URL
    bucket_name = settings.MDS_S3_AVIA_ADMIN_BUCKET

    log.info(
        'Launching s3 sync for config %s. S3 endpoint %s',
        config,
        endpoint_url,
    )

    s3 = boto3.resource('s3', endpoint_url=endpoint_url)
    bucket = s3.Bucket(bucket_name)

    try:
        if not bucket_exists(s3, bucket):
            log.info('Bucket does not exist, creating new bucket %s', bucket_name)
            s3.create_bucket(
                Bucket=bucket_name,
                CreateBucketConfiguration={
                    'LocationConstraint': 'us-east-1',
                },
            )
        else:
            log.debug('Bucket %s was found in storage', bucket)
    except Exception as e:
        msg = (
            'Connection failed (%s). Please check for correct S3 credentials in environment variables '
            '(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY) for host %s'
        )
        log.exception(msg, e, endpoint_url)
        raise S3SyncError(msg % (e, endpoint_url))

    try:
        sync_list = settings.S3_SYNC_CONFIGS[config]
    except KeyError:
        msg = 'No configuration "%s" for s3_sync, choose one of %s (see settings.S3_SYNC_CONFIGS)'
        log.error(msg, config, [key for key in settings.S3_SYNC_CONFIGS.keys()])
        raise S3SyncError(msg % config, [key for key in settings.S3_SYNC_CONFIGS.keys()])

    for share, path, exclude in sync_list:
        try:
            log.info('cp %s => [%s]/%s', path, bucket, share)
            s3_cp(bucket, share, path, excludes=exclude)
        except Exception as e:
            msg = 'Could not copy %s to [%s]/%s (%s)'
            log.exception(msg, path, bucket, share, e)
            raise S3SyncError(msg % (path, bucket, share, e))


def s3_cp(bucket, s3_key_prefix, path, excludes=None):
    dirname = os.path.dirname(path)
    basename = os.path.basename(path)
    s3_copy_recursive(bucket, s3_key_prefix, dirname, basename, excludes)


def s3_copy_recursive(bucket, s3_key_prefix, path_prefix, path, excludes=None):
    realpath = os.path.join(path_prefix, path)
    if excludes is None:
        excludes = []
    for exc in excludes:
        if fnmatch.fnmatch(realpath, exc):
            log.debug('File %s is excluded', realpath)
            return
    if not os.path.exists(realpath):
        log.error('File %s not found', realpath)
        return
    if os.path.isdir(realpath):
        log.debug('File %s is a directory', realpath)
        for f in os.listdir(realpath):
            s3_copy_recursive(bucket, s3_key_prefix, path_prefix, os.path.join(path, f))
        return

    s3_copy_file(bucket, s3_key_prefix, path_prefix, path)


def s3_copy_file(bucket, s3_key_prefix, path_prefix, path):
    realpath = os.path.join(path_prefix, path)
    s3_key = os.path.join(s3_key_prefix, path)
    log.debug('cp %s -> [%s]/%s', realpath, bucket, s3_key)
    try:
        mime = magic.from_file(realpath, mime=True)
        bucket.Object(s3_key).upload_file(
            realpath,
            ExtraArgs={
                'ContentType': mime,
            }
        )
    except magic.MagicException:
        log.exception('Could not detect MIME type for file: %s', realpath)
        bucket.Object(s3_key).upload_file(realpath)
