# -*- coding: utf8 -*-
import os

from django.conf import settings
from django.utils.html import escape
from jinja2 import Template

from travel.avia.library.python.common.lib.mail import SmailikClient
from travel.avia.library.python.common.models.scripts import Script, ScriptResult


MAIL_TEMPLATE = u"""<b>Логи:</b><br />
{{logs|e}}
<br />
<br />
<b>Перейти к:</b><br />
1) <a href="{{script_url}}">Станица скрипта</a><br />
2) <a href="{{script_result_url}}">Страница со статусом выполениня скрипта</a><br />"""
MAX_LOG_SIZE = 4096


class FailScriptMailSender(object):
    """
    Рассыльщик писем о падении одного конкретного скрипта
    """
    def __init__(self, mail_sender):
        # type: (SmailikClient) -> None
        self._mail_sender = mail_sender
        self._template = Template(MAIL_TEMPLATE)

    def send(self, script, script_result, to_email=None):
        # type: (Script, ScriptResult) -> None
        self._mail_sender.send_email(
            to_email or settings.SCRIPT_REPORTER_MAIL,
            escape(self._create_subject(script, script_result)),
            self._create_message(script, script_result)
        )

    def _create_subject(self, script, script_result):
        # type: (Script, ScriptResult) -> unicode
        return u'Админка: упал скрипт [{code}] в {when}.'.format(
            code=script.code,
            when=script_result.started_at.strftime('%Y-%m-%d %H:%M')
        )

    def _create_message(self, script, script_result):
        # type: (Script, ScriptResult) -> unicode
        return self._template.render(
            script_url=u"{}/admin/www/script/{}/".format(settings.HOST_URL, script.id),
            script_result_url=u"{}/admin/www/scriptresult/{}/".format(settings.HOST_URL, script_result.id),
            logs=self._read_logs(script_result)
        )

    def _read_logs(self, script_result):
        # type: (ScriptResult) -> unicode
        log_file_path = script_result.log_file
        if not log_file_path:
            return u'Нет логов - нет проблем'
        if not os.path.isfile(log_file_path):
            return u'Нет логов - нет проблем'

        with open(log_file_path) as log_file:
            message = log_file.read()
            if len(message) > MAX_LOG_SIZE:
                message = '...' + message[-MAX_LOG_SIZE:]
            return message


fail_script_mail_sender = FailScriptMailSender(
    mail_sender=SmailikClient(
        service='Flights.Yandex',
        send_from=u'Яндекс.Авиабилеты &lt;noreply@admin.avia.yandex-team.ru&gt;',
        smailik_dir='/var/spool/smailik-preparer/avia-scripts/'
    )
)
