# -*- coding: utf8 -*-
from django.conf import settings
from django.utils.html import escape
from jinja2 import Template
from typing import Union

from travel.avia.library.python.common.lib.mail import SmailikClient

from travel.avia.admin.lib.script_helpers.script_reporter import script_reporter, ScriptReporter

MAIL_TEMPLATE = u"""<b>Краткий отчет:</b><br />
{%- if report.daily_failed_scripts %}
За сегодня упали следующие скрипты: <b>{{report.daily_failed_scripts|join(', ', attribute='code') }}</b>.
{%- elif report.weekly_failed_scripts %}
За неделю упали следующие скрипты: <b>{{report.weekly_failed_scripts|join(', ', attribute='code') }}</b>.
{%- else %}
Инцидентов <b>нет</b>.
{%- endif %}
<br />
<br />
<b>Полный отчет:</b>
<br />
<table style="width:100%; {{styles}}">
    <thead style="{{styles}}">
        <tr style="{{styles}}">
            <th rowspan="2" style="{{styles}}">Скрипт</th>
            <th colspan="4" style="{{styles}}">Дневной</th>
            <th colspan="4" style="{{styles}}">Недельный</th>
        </tr>
        <tr style="{{styles}}">
            <th style="{{styles}}">Ok</th>
            <th style="{{styles}}">Error</th>
            <th style="{{styles}}">Last Ok</th>
            <th style="{{styles}}">Last Error</th>
            <th style="{{styles}}">Ok</th>
            <th style="{{styles}}">Error</th>
            <th style="{{styles}}">Last Ok</th>
            <th style="{{styles}}">Last Error</th>
        </tr>
    </thead>
    <tbody style="{{styles}}">
    {%- for script_report in report.script_reports %}
    <tr style="{{styles}}">
            <td style="text-align: center; {{styles}}">
                <a href="{{admin_host}}/admin/www/script/{{script_report.script.id}}/">
                    {{script_report.script.code}}
                </a>
            </td>
            {%- set sub_reports = (script_report.daily, script_report.weekly) %}
            {%- for sub_report in sub_reports %}
            <td style="text-align: center; {{styles}}">
                <b style="color: #039660;">{{'%d' % sub_report.count_success_runs}}</b>
            </td>
            <td style="text-align: center; {{styles}}">
                <b style="color: #f00;">{{'%d' % sub_report.count_error_runs}}</b>
            </td>
            <td style="text-align: center; {{styles}}">
                {%- if sub_report.last_error_run %}
                <b style="color: #039660;">
                    <a href="{{admin_host}}/admin/www/scriptresult/{{sub_report.last_error_run.id}}/">
                        {{sub_report.last_error_run.started_at.strftime('%m-%dT%H:%M')}}
                    </a>
                </b>
                {%- else %}
                    -
                {%- endif %}
            </td>
            <td style="text-align: center; {{styles}}">
                {%- if sub_report.last_success_run %}
                <b style="color: #f00;">
                    <a href="{{admin_host}}/admin/www/scriptresult/{{sub_report.last_success_run.id}}/">
                        {{sub_report.last_success_run.started_at.strftime('%m-%dT%H:%M')}}
                    </a>
                </b>
                {%- else %}
                -
                {%- endif %}
            </td>
            {%- endfor %}
    </tr>
  {%- endfor %}
  </tbody>
</table>
"""


class ScriptReportsMailSender(object):
    def __init__(self, script_reporter, mail_sender):
        # type: (ScriptReporter, SmailikClient) -> None
        self._script_reporter = script_reporter
        self._mail_sender = mail_sender
        self._template = Template(MAIL_TEMPLATE)

    def send(self, to_email=None):
        # type: (Union[str, None]) -> None
        report = self._script_reporter.make()

        subject = self._create_subject(report)
        message = self._create_message(report)

        self._mail_sender.send_email(
            to_email or settings.SCRIPT_REPORTER_MAIL,
            escape(subject),
            message
        )

    def _create_subject(self, report):
        subject = u'Админка: скрипты работают.'
        if report.weekly_failed_scripts > 0:
            subject = u'Админка: в течении недели падали скрипты.'
        if report.daily_failed_scripts > 0:
            subject = u'Админка: в течении дня падали скрипты.'

        return subject

    def _create_message(self, report):
        return self._template.render(
            styles=u'border: 1px solid black;border-collapse: collapse;',
            report=report,
            admin_host=settings.HOST_URL
        )


script_reports_mail_sender = ScriptReportsMailSender(
    script_reporter=script_reporter,
    mail_sender=SmailikClient(
        service='Flights.Yandex',
        send_from=u'Яндекс.Авиабилеты &lt;noreply@admin.avia.yandex-team.ru&gt;',
        smailik_dir='/var/spool/smailik-preparer/avia-scripts/'
    )
)
