from collections import defaultdict
from datetime import timedelta, datetime

from typing import Union, Tuple

from travel.avia.library.python.common.models.scripts import Script, ScriptResult
from travel.avia.library.python.common.utils import environment


class PeriodScriptReport(object):
    def __init__(self, last_error_run, last_success_run,
                 count_error_runs, count_success_runs):
        # type: (Union[ScriptResult, None], Union[ScriptResult, None], int, int) -> None
        self.last_error_run = last_error_run
        self.last_success_run = last_success_run

        self.count_error_runs = count_error_runs
        self.count_success_runs = count_success_runs

    def __json__(self):
        return {
            'last_error_run': self.last_error_run.started_at.strftime('%Y-%m-%dT%H:%M') if self.last_error_run else None,
            'last_success_run': self.last_success_run.started_at.strftime('%Y-%m-%dT%H:%M') if self.last_success_run else None,
            'count_error_runs': self.count_error_runs,
            'count_success_runs': self.count_success_runs,
        }


class ScriptReport(object):
    def __init__(self, script, daily, weekly, monthly, total):
        # type: (Script, PeriodScriptReport, PeriodScriptReport, PeriodScriptReport, PeriodScriptReport) -> None
        self.script = script

        self.daily = daily
        self.weekly = weekly
        self.monthly = monthly
        self.total = total

    def __json__(self):
        return {
            'script': self.script.code,
            'daily': self.daily.__json__(),
            'weekly': self.weekly.__json__(),
            'monthly': self.monthly.__json__(),
            'total': self.total.__json__()
        }

    def __repr__(self):
        return 'ScriptReport[{}]'.format(self.script.code)


class Report(object):
    def __init__(self, daily_failed_scripts, weekly_failed_scripts,
                 monthly_failed_scripts, script_reports):
        # type: (Tuple[Script, ...], Tuple[Script, ...], Tuple[Script, ...], Tuple[ScriptReport, ...]) -> None
        self.daily_failed_scripts = daily_failed_scripts
        self.weekly_failed_scripts = weekly_failed_scripts
        self.monthly_failed_scripts = monthly_failed_scripts

        self.script_reports = script_reports

    def __json__(self):
        return {
            'daily_failed_scripts': [s.code for s in self.daily_failed_scripts],
            'weekly_failed_scripts': [s.code for s in self.daily_failed_scripts],
            'monthly_failed_scripts': [s.code for s in self.daily_failed_scripts],
            'script_reports': [r.__json__() for r in self.script_reports]
        }


class ScriptReporter(object):
    def __init__(self, environment):
        self._environment = environment

    def make(self):
        # type: () -> Report
        scripts = list(Script.objects.all().order_by('code'))
        script_results = list(ScriptResult.objects.all().order_by('-started_at'))

        result_by_script = defaultdict(list)
        for result in script_results:
            result_by_script[result.script_id].append(result)

        reports = tuple(self._report_for_script(s, result_by_script[s.id]) for s in scripts)

        def get_failed_scripts(sub_report_name):
            return tuple(r.script for r in reports if getattr(r, sub_report_name).count_error_runs)

        return Report(
            daily_failed_scripts=get_failed_scripts('daily'),
            weekly_failed_scripts=get_failed_scripts('weekly'),
            monthly_failed_scripts=get_failed_scripts('monthly'),
            script_reports=reports
        )

    def _report_for_script(self, script, results):
        # type: (Script, Tuple[ScriptResult, ...]) -> ScriptReport
        now = self._environment.now()

        return ScriptReport(
            script=script,

            daily=self._range_report_for_script(
                max_date=now - timedelta(days=1),
                results=results
            ),
            weekly=self._range_report_for_script(
                max_date=now - timedelta(days=7),
                results=results
            ),
            monthly=self._range_report_for_script(
                max_date=now - timedelta(days=30),
                results=results
            ),
            total=self._range_report_for_script(
                max_date=None,
                results=results
            )
        )

    def _range_report_for_script(self, max_date, results):
        # type: (Union[datetime, None], Tuple[ScriptResult, ...]) -> PeriodScriptReport
        last_success_run = None
        last_error_run = None
        count_error_runs = 0
        count_success_runs = 0

        for r in results:
            if max_date and max_date > r.started_at:
                break

            is_success = r.success
            is_fail = not r.success and r.finished_at

            if is_success:
                if not last_success_run:
                    last_success_run = r
                count_success_runs += 1

            if is_fail:
                if not last_error_run:
                    last_error_run = r
                count_error_runs += 1

        return PeriodScriptReport(
            last_error_run=last_error_run,
            last_success_run=last_success_run,

            count_error_runs=count_error_runs,
            count_success_runs=count_success_runs
        )


script_reporter = ScriptReporter(environment)
