# -*- coding: utf8 -*-
import subprocess
import timeout_decorator
import logging
from typing import Callable

from django.conf import settings

from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.models.scripts import Script, ScriptResult

from travel.avia.admin.lib.script_helpers.environment_detector import EnvironmentDetector, environment_detector
from travel.avia.admin.lib.script_helpers.script_run_logger_factory import ScriptRunLoggerFactory, script_run_logger_factory
from travel.avia.admin.lib.script_helpers.mail_senders.fail_script_mail_sender import FailScriptMailSender, fail_script_mail_sender

DAY_IN_SECONDS = 60 * 60 * 24 * 1


class ScriptRunner(object):
    def __init__(self, environment_detector, script_run_logger_factory,
                 subprocess, environment, logger, fail_script_mail_sender):
        # type: (EnvironmentDetector, ScriptRunLoggerFactory, any, any, logging.Logger, FailScriptMailSender) -> None
        self._environment_detector = environment_detector
        self._script_run_logger_factory = script_run_logger_factory
        self._subprocess = subprocess
        self._environment = environment
        self._fail_script_mail_sender = fail_script_mail_sender
        self._logger = logger

    def _create_command(self, script):
        # type: (Script) -> str
        command = script.command.strip().format(
            project_root=settings.PROJECT_PATH
        )

        # Хак для Аркадии, чтобы много не переписывать.
        # Код подойдет для любого скрипта из avia_scripts,
        # поэтому при необходимости можно будет сделать список скриптов.
        if script.code == 'build_similar_airlines':
            args_start_index = command.find(script.code)
            if args_start_index == -1:
                args = ''
            else:
                args_start_index += len(script.code) + len('.py ')
                args = command[args_start_index:]
            return 'Y_PYTHON_ENTRY_POINT=travel.avia.admin.avia_scripts.{code}:main /usr/bin/timeout -k 60 -s 9 {timeout} zk-flock {code} "{app_binary} {args} --script-code {code}"'.format(
                timeout=script.timeout,
                args=args,
                code=script.code,
                app_binary=settings.APP_BINARY,
            )

        return '/usr/bin/timeout -k 60 -s 9 {timeout} zk-flock {code} "{command} --script-code {code}"'.format(
            timeout=script.timeout,
            command=command,
            code=script.code
        )

    @timeout_decorator.timeout(DAY_IN_SECONDS)
    def run_in_another_process(self, code):
        # type: (str) -> (bool, str)
        self._logger.info('Start: %s', code)
        script = Script.objects.filter(code=code).first()
        if not script:
            message = 'script [{}] is unregistered'.format(code)
            self._logger.error(message)
            return False, message

        environment_type = self._environment_detector.get_environment_type()
        if not getattr(script, 'enabled_in_{}'.format(environment_type)):
            message = 'script [{}] is disabled in {} environment'.format(code, environment_type)
            self._logger.info(message)
            return False, message

        if not script.command:
            message = 'script [{}] is not executable'.format(code)
            self._logger.error(message)
            return False, message

        try:
            command = self._create_command(script)
        except Exception as e:
            self._logger.exception(e)
            return False, 'script [{}] has broken command [{}]'.format(script.code, script.command)

        self._logger.info('Run command: %s', command)

        try:
            return_code = self._subprocess.call(command, shell=True)
        except Exception as e:
            self._logger.exception(e)
            return False, 'subprocess throw exception [{}], when executing script [{}]'.format(e, script.code)

        if return_code != 0:
            message = 'script [{}] return unexpected code [{}]'.format(script.code, return_code)
            self._logger.error(message)
            return False, message

        self._logger.info('Finish: %s', code)

        return True, ''

    @timeout_decorator.timeout(DAY_IN_SECONDS)
    def run_in_process(self, handler, script_code=None):
        if script_code is None:
            import argparse
            parser = argparse.ArgumentParser()
            parser.add_argument('--script-code', dest='script_code')
            args = parser.parse_args()

            script_code = args.script_code

        try:
            self._run_in_process(handler, script_code)
        except Exception as e:
            self._logger.exception(e)
            self._logger.info('Fail: %s', script_code)
            raise e

    def _run_in_process(self, handler, script_code):
        # type: (Callable, str) -> None
        script = Script.objects.filter(code=script_code).first()
        if not script:
            raise Exception('Can\'t execute script: {}'.format(script_code))

        path_to_file_log = script_run_logger_factory.create(script.code)
        self._logger.info('Start: %s', script_code)

        script_result = ScriptResult(
            script=script,
            started_at=self._environment.now(),
            log_file=path_to_file_log
        )
        script_result.save()

        handler_error = None
        try:
            handler()
        except Exception as e:
            handler_error = e

        script_result.finished_at = self._environment.now()
        script_result.success = handler_error is None
        script_result.save()

        if not script_result.success:
            self._fail_script_mail_sender.send(script, script_result)
            raise handler_error
        self._logger.info('End: %s', script_code)


script_runner = ScriptRunner(
    environment_detector=environment_detector,
    script_run_logger_factory=script_run_logger_factory,
    subprocess=subprocess,
    environment=environment,
    logger=logging.getLogger(__name__),
    fail_script_mail_sender=fail_script_mail_sender
)
