import logging
import time
from urlparse import urljoin

import requests
from django.conf import settings
from retrying import retry

_headers = {
    'user-agent': 'avia-admin',
    'content-type': 'application/json',
}


class SimpleDaemonClient(object):
    def __init__(self, logger):
        self._logger = logger

    @retry(stop_max_attempt_number=3)
    def init(self, params):
        init_url = urljoin(
            settings.TICKET_DAEMON_API_URL, 'jsendapi/front/init_search_by_query/'
        )
        self._logger.info('init query: [%s] [%r]', init_url, params)

        r = requests.get(
            init_url,
            params=params,
            headers=_headers
        )
        r.raise_for_status()

        return r.json()['data']['id']

    @retry(stop_max_attempt_number=3)
    def results(self, qid, cont=0):
        results_url = urljoin(
            settings.TICKET_DAEMON_API_URL,
            'jsendapi/v3/results/{}/{}'.format(qid, cont)
        )

        self._logger.info('fetch results: [%s]', results_url)

        r = requests.get(
            results_url,
            params={'ignore_outdated': True},
            headers=_headers
        )
        r.raise_for_status()

        data = r.json()['data']

        cont = data['cont']
        fares = data['variants']['fares']

        return cont, fares

    def fetch(self, params):
        qid = self.init(params)
        cont = 0
        fares = []

        for i in range(5):
            time.sleep(5)
            cont, new_fares = self.results(qid, cont)
            fares += new_fares
            if cont is None:
                break

        return fares

    def find_min_price(self, params, base_currency):
        self._logger.info('find min price: [%r]', params)
        fares = self.fetch(params)

        min_price = None

        for fare in fares:
            for p in fare['prices']:
                t = p['tariff']
                value = t['value']
                currency = t['currency']

                if currency != base_currency:
                    continue

                if not min_price or min_price > value:
                    min_price = value

        return min_price


simple_daemon_client = SimpleDaemonClient(logging.getLogger(__name__))
