# -*- coding: utf-8 -*-

import logging
from collections import OrderedDict
from datetime import datetime, time

import xlrd
from xlrd.xldate import XLDateAmbiguous


log = logging.getLogger(__name__)


class XlsParser(object):
    def __init__(self, filepath, transform_to_text=False, strip_values=False):
        self.strip_values = strip_values
        self.filepath = filepath
        self.work_book = xlrd.open_workbook(self.filepath)
        self.sheet = self.work_book.sheet_by_index(0)
        self.row_number = 0

        self.transform_to_text = transform_to_text

    def __iter__(self):
        return self

    def next(self):
        if self.row_number < self.sheet.nrows:
            if not self.transform_to_text:
                row = self.sheet.row_values(self.row_number)
            else:
                row = self.get_transformed_row()

            self.row_number += 1

            if self.strip_values:
                row = tuple(map(lambda x: x.strip() if isinstance(x, basestring) else x, row))

            return row
        else:
            raise StopIteration()

    def get_transformed_row(self):
        row = []
        for collx in range(0, self.sheet.ncols):
            row.append(self.transform_value_to_text(self.row_number, collx))

        return tuple(row)

    def transform_value_to_text(self, rowx, collx):
        value = self.sheet.cell_value(rowx, collx)
        type_ = self.sheet.cell_type(rowx, collx)

        if type_ == xlrd.XL_CELL_DATE:
            try:
                year, month, day, hours, minutes, seconds = xlrd.xldate_as_tuple(value, self.work_book.datemode)
            except XLDateAmbiguous:
                year, month, day, hours, minutes, seconds = xlrd.xldate_as_tuple(value, 1)

                time_ = unicode(time(hours, minutes, seconds).strftime("%H:%M:%S"))
                log.error(u"Ошибка разбора даты времени cell(%s, %s) %s %s, принудительно преобразуев во время %s",
                          rowx, collx, value, self.work_book.datemode, time_)

                return time_

            # it is time type
            if year == 0:
                return unicode(time(hours, minutes, seconds).strftime("%H:%M:%S"))
            else:
                return unicode(datetime(year, month, day, hours, minutes, seconds).strftime("%Y-%m-%d %H:%M:%S"))

        elif type_ == xlrd.XL_CELL_NUMBER:
            if int(value) == value:
                return unicode(int(value))
            else:
                return unicode(value)

        else:
            return unicode(value)


class XlsDictParser(XlsParser):
    def __init__(self, filepath, transform_to_text=False, fieldnames=None,
                 strip_values=False, tail_field_name='tail'):
        super(XlsDictParser, self).__init__(filepath, transform_to_text, strip_values=strip_values)

        if not fieldnames:
            fieldnames = super(XlsDictParser, self).next()

        self.fieldnames = fieldnames
        self.tail_field_name = tail_field_name

    def __iter__(self):
        return self

    def next(self):
        tail = None
        rowdict = OrderedDict()
        row = super(XlsDictParser, self).next()

        if len(row) > len(self.fieldnames):
            tail = row[len(self.fieldnames):]
            row = row[:len(self.fieldnames)]
        elif len(row) < len(self.fieldnames):
            row = row + (None,)*(len(self.fieldnames) - len(row))

        for key, value in zip(self.fieldnames, row):
            rowdict[key] = value

        if tail is not None:
            rowdict[self.tail_field_name] = tail

        return rowdict
