# -*- coding: utf-8 -*-
import csv
import os
import uuid

from contextlib import contextmanager
from datetime import date, datetime, timedelta

from django.conf import settings

import yt.wrapper as yt
from yt.wrapper import YtClient


@contextmanager
def temp_table(yt):
    table = yt.create_temp_table(settings.YT_TMP_DIRECTORY)
    try:
        yield table
    finally:
        yt.remove(table)


def create_safe_temp_table(prefix, path='//home/avia/tmp/', attributes=None, ytc=yt):
    """
    Создает временную таблицу в `path`
    (по мотивам https://st.yandex-team.ru/YTADMINREQ-6326)
    Удалением занимается вызывающая сторона
    """

    if prefix[0] == '/':
        # Делаем из абослютного пути относительный
        prefix = prefix[1:]

    table_name = os.path.join(path, prefix + '-' + str(uuid.uuid4()))
    ytc.create('table', table_name, recursive=True, attributes=attributes)

    return table_name


def configure_wrapper(yt):
    yt.config["proxy"]["url"] = settings.YT_PROXY
    yt.config["token"] = settings.YT_TOKEN
    yt.config["read_retries"]["enable"] = True
    yt.config["clear_local_temp_files"] = True
    yt.config["remote_temp_tables_directory"] = settings.YT_TMP_DIRECTORY


def last_logs_tables(yt, path, days):
    today = datetime.now().date()
    return safe_tables_for_daterange(yt, path, today-timedelta(days), today)


def _table_date(table, date_format='%Y-%m-%d'):
    return datetime.strptime(table.split('/')[-1], date_format).date()


def tables_for_daterange(yt, path, begin, end, date_format='%Y-%m-%d'):
    return [
        t for t in yt.search(path, node_type='table')
        if begin <= _table_date(t, date_format) <= end
    ]


def safe_tables_for_daterange(yt, path, begin, end, date_format='%Y-%m-%d'):
    tables = []
    for t in yt.search(path, node_type='table'):
        try:
            if begin <= _table_date(t, date_format) <= end:
                tables.append(t)
        except:
            continue

    return tables


def safe_dates_and_tables_for_daterange(yt, path, begin, end, date_format='%Y-%m-%d', include_end=True):
    dates_and_tables = []
    in_daterange = (lambda x: begin <= x <= end) if include_end else (lambda x: begin <= x < end)
    for t in yt.search(path, node_type='table'):
        try:
            day = _table_date(t, date_format)
            if in_daterange(day):
                dates_and_tables.append((day, t))
        except Exception:
            continue

    return dates_and_tables


def yt_last_logs_tables(path, days, skip_today=False):
    tables = []
    today = date.today()

    for table in yt.search(path, node_type="table"):
        try:
            table_date = datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()
        except ValueError:
            continue

        if skip_today and table_date == today:
            continue

        tables.append((
            table_date,
            table
        ))

    sorded_tables = sorted(tables, reverse=True)[:days]

    return [table for _table_date, table in sorded_tables]


def download_table_to_csv(table_path, outp, columns, need_header, delimiter,
                          start_index=None, end_index=None):
    if need_header:
        writer = csv.DictWriter(outp, fieldnames=columns, delimiter=delimiter)
        writer.writeheader()

    table_path = yt.TablePath(
        name=table_path,
        start_index=start_index,
        end_index=end_index,
    )

    format = yt.SchemafulDsvFormat(
        columns=columns,
        attributes={"field_separator": delimiter}
    )
    for record in yt.read_table(table_path, format=format, raw=True):
        outp.write(record)


def table_is_empty(ytc, table_path):
    """

    :type table_path: basestring
    :type ytc: YtClient
    :rtype: bool
    """
    table = ytc.TablePath(
        name=table_path,
        start_index=0,
        end_index=1,
    )
    for _ in ytc.read_table(table):
        return False
    return True


class YtClientFabric(object):
    def __init__(self):
        self._instance = None

    def create(self):
        if not self._instance:
            self._instance = YtClient(
                proxy=settings.YT_PROXY,
                token=settings.YT_TOKEN,
                config={
                    'operation_tracker': {
                        'stderr_logging_level': 'WARNING',
                        'progress_logging_level': 'WARNING',
                    },
                    'remote_temp_tables_directory': settings.YT_TMP_DIRECTORY,
                },
            )

        return self._instance


yt_client_fabric = YtClientFabric()


class YtTableProvider(object):
    def __init__(self, yt_fabric, environment):
        self._yt_fabric = yt_fabric
        self._environment = environment

    def get_last(self, path, days, skip_today=False):
        tables = []
        today = self._environment.today()

        for table in self._yt_fabric.create().search(path, node_type="table"):
            try:
                table_date = datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()
            except ValueError:
                continue

            if skip_today and table_date == today:
                continue

            tables.append((
                table_date,
                table
            ))

        sorted_tables = sorted(tables, reverse=True)[:days]

        return [table for _table_date, table in sorted_tables]


yt_table_provider = YtTableProvider(
    yt_fabric=yt_client_fabric,
    environment=date
)


def yt_read_tables(yt_client, tables, columns=None, format=yt.JsonFormat()):
    table_generator = (yt.TablePath(table, columns=columns) for table in tables)
    for table in table_generator:
        for record in yt_client.read_table(table, format=format):
            yield record


class AviaYtClient(YtClient):
    def tables_for_daterange(self, path, left_date, right_date):
        return safe_tables_for_daterange(self, path, left_date, right_date)

    def read_tables(self, tables, columns=None, format=yt.JsonFormat()):
        for r in yt_read_tables(self, tables, columns, format):
            yield r


class AviaYtClientFabric(object):
    def create(self):
        return AviaYtClient(
            proxy=settings.YT_PROXY,
            token=settings.YT_TOKEN,
            config={
                'operation_tracker': {
                    'stderr_logging_level': 'WARNING',
                    'progress_logging_level': 'WARNING',
                },
                'remote_temp_tables_directory': settings.YT_TMP_DIRECTORY,
            },
        )
