# -*- coding:utf-8 -*-
import hashlib
import logging
from datetime import datetime

import yt.wrapper as yt
from django.conf import settings
from django.core.exceptions import ValidationError
from django.http import HttpResponse, HttpResponseBadRequest, HttpResponseServerError

from travel.avia.library.python.common.models.partner import Partner
from travel.avia.admin.lib import yt_helpers as yth
from travel.avia.admin.lib.marker import MarkerWriter, MarkerTransfer, BaseMarkerReader

logger = logging.getLogger(__name__)

SOURCE = 'tripcom'
PARTNER = 'tripcom'


class CtripMarkerReader(BaseMarkerReader):
    def __init__(self, raw_data):
        """
        :param django.http.request.QueryDict raw_data:
        """
        super(CtripMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'FLIGHT_WAITING_PAY': 'booking',
                'FLIGHT_WAITING_CHARGE': 'booking',
                'FLIGHT_PAYED_TICKETING': 'paid',
                'FLIGHT_CHARGED_TICKETING': 'paid',
                'FLIGHT_TICKETING': 'paid',
                'FLIGHT_TICKETED': 'paid',
                'FLIGHT_UNCOMMIT': 'cancel',
                'FLIGHT_UNSUBCRIBE_PART': 'cancel',
                'FLIGHT_UNSUBCRIBE_ALL': 'cancel',
                'FLIGHT_CANCELLING': 'cancel',
                'FLIGHT_CANCELLED': 'cancel',
            },
        )
        self.raw_data = raw_data

    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат структуры смотри в parse_report

        :param datetime.datetime date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        return self.parse_report(self.raw_data)

    def parse_report(self, content):
        """
        Обрабатываем данные от партнера и возвращаем массив заказов со следующей структурой
        [{
            order_id: ...,

            created_at: ...,  --datetime format

            price: ...,

            status: ...,  (booking, paid, cancel)

            marker: ...,

        }]

        :param django.http.request.QueryDict content: Данные GET запроса от ctrip
        :return: массив с заказами из одного элемента.
        Возвращаем массивом для соответствия установленным ранее контрактам.
        :rtype: list[dict]
        """
        orders = []
        order = {
            'order_id': content.get('orderid'),
            'created_at': self._convert_dt(content.get('orderdate')),
            'price': content.get('orderamount'),
            'status': self._parse_status(content.get('orderstatus')),
            'marker': content.get('ouid'),
        }
        orders.append(order)
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')

    def validate_token(self):
        if hashlib.md5(''.join([
            self.raw_data['allianceid'],
            self.raw_data['sid'],
            self.raw_data['orderid'],
        ])).hexdigest().upper() != self.raw_data['token']:
            raise ValidationError('token invalid')

    def get_order_date(self):
        return self._convert_dt(self.raw_data['orderdate'])


class DummyWriter(MarkerWriter):
    def write_to_yt(self, date):
        self._logger.info("Ctrip order recieved for date %s : %s", date, self._rows)


def _create_writer(current_env):
    allowed_yt_envs = ['production']
    yth.configure_wrapper(yt)
    if current_env in allowed_yt_envs:
        marker_writer = MarkerWriter(SOURCE, logger)
    else:
        logger.info(
            'Current ENVIRONMENT %s. Dry run. Run in %s to write to YT.',
            current_env,
            ', '.join(allowed_yt_envs)
        )
        marker_writer = DummyWriter(SOURCE, logger)

    return marker_writer


def process_data(ctrip_data):
    marker_writer = _create_writer(settings.ENVIRONMENT)
    marker_reader = CtripMarkerReader(ctrip_data)
    marker_reader.validate_token()

    marker_transfer = MarkerTransfer(
        partner=Partner.objects.get(code=PARTNER),
        marker_writer=marker_writer,
        marker_reader=marker_reader,
        logger=logger,
    )
    report_date = marker_reader.get_order_date()
    logger.info('Transfering order for date: %s', report_date)
    # стоит сделать асинхронно, если будет много запросов и тормоза
    marker_transfer.transfer(report_date)
    logger.info('Stop: transfering data')


def postback_api(request):
    try:
        # Убрать когда будет понятно, что всё работает
        logger.info('Recieveing data from Ctrip: %s', request.GET)
        process_data(request.GET)
        return HttpResponse('ok')
    except ValidationError:
        logger.exception('Ctrip invalid data')
        return HttpResponseBadRequest()
    except Exception:
        logger.exception('Ctrip data transfer error')
        return HttpResponseServerError()
