# -*- coding: utf-8 -*-
import heapq
import json
import operator
import re
import string
import time as os_time
from functools import update_wrapper
from traceback import format_exc
from urlparse import urljoin

from datetime import datetime, timedelta

import requests
from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.admin import SimpleListFilter, ModelAdmin
from django.contrib.admin.options import csrf_protect_m, get_content_type_for_model
from django.contrib.admin.utils import unquote, flatten_fieldsets
from django.contrib.admin.widgets import AdminDateWidget, AdminFileWidget, ForeignKeyRawIdWidget
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ObjectDoesNotExist, PermissionDenied
from django.db import transaction
from django.db.models.fields.related import ManyToOneRel
from django.http import HttpResponseRedirect
from django.template.response import TemplateResponse
from django.utils.decorators import method_decorator
from django.utils.encoding import force_text
from django.utils.functional import curry
from django.utils.http import urlencode
from django.utils.safestring import mark_safe
from django.utils.text import capfirst
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.avia_data.models import AmadeusMerchant
from travel.avia.library.python.common.models.base_partner_model import DEFAULT_CACHE_TTL
from travel.avia.library.python.common.models.geo import Point, Settlement, Station
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models.partner import (
    BlablacarToken, ClickPriceMultiplierByRedirectType, EmailType, DohopIndex, DohopVendor, DohopVendorIndexInfo, Partner,
    PartnerEmail, RegionalizePartnerQueryRule, PartnerUser,
    PriceList, DefaultClickPrice, CPCPrice,
    ConversionByRedirectType, RedirectType,
    PartnerVariantsTTL, DohopVariantsTTL, AmadeusMerchantVariantsTTL, UpdateHistoryRecord
)
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.httpresponses import jsonp_response
from travel.avia.library.python.common.utils.media_fields import SvgImageField, CustomImageField
from travel.avia.library.python.common.utils.safe_xml_parser import safe_parse_xml
from travel.avia.admin.lib.admin_options import RaspExportModelAdmin
from travel.avia.admin.order.models import CoachInfoBinding, CoachInfo, ServiceClass, CoachService, QueryBlackList
from travel.avia.admin.xgettext.i18n import xngettext, gettext


jsonp_response_m = method_decorator(jsonp_response)
MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES = 1 * 1024 * 1024


class CoachInfoBindingInline(admin.TabularInline):
    model = CoachInfoBinding


class CPCPriceInline(admin.TabularInline):
    model = CPCPrice


class CoachInfoAdmin(RaspExportModelAdmin):
    list_display = ('name',)
    inlines = [
        CoachInfoBindingInline,
    ]
    fieldsets = (
        ('Profile field', {
            'fields': ('name', 'image', 'schema', 'seat_selection_algo')
        }),
    )

admin.site.register(CoachInfo, CoachInfoAdmin)


class CoachInfoBindingAdmin(RaspExportModelAdmin):
    list_display = ('klass', 'service_class', 'train_number', 'coach_number', 'priority', 'info', 'two_storey')

admin.site.register(CoachInfoBinding, CoachInfoBindingAdmin)


class ServiceClassAdmin(RaspExportModelAdmin):
    filter_horizontal = ('services',)
    list_display = ('name', 'owner', 'code', 'is_brand', 'note')

admin.site.register(ServiceClass, ServiceClassAdmin)


class PartnerEmailAdminInline(admin.TabularInline):
    model = PartnerEmail


class BaseVariantsTTLAdminInline(admin.TabularInline):
    DEFAULT_TTL_CHANGE_DEPTH = 3

    def get_formset(self, request, obj=None, **kwargs):
        """Add initial rows when creating partners. Initial value must be empty when arriving POST request"""
        initial = []
        if 'add' in request.path.split('/') and request.method == 'GET':
            initial = [
                {'search_depth': self.DEFAULT_TTL_CHANGE_DEPTH, 'ttl': DEFAULT_CACHE_TTL},
                {'ttl': DEFAULT_CACHE_TTL},
            ]
        formset = super(BaseVariantsTTLAdminInline, self).get_formset(request, obj, **kwargs)
        formset.__init__ = curry(formset.__init__, initial=initial)
        return formset


class PartnerVariantsTTLAdminInline(BaseVariantsTTLAdminInline):
    model = PartnerVariantsTTL


class DohopVariantsTTLAdminInline(BaseVariantsTTLAdminInline):
    model = DohopVariantsTTL


class AmadeusMerchantVariantsTTLAdminInline(BaseVariantsTTLAdminInline):
    model = AmadeusMerchantVariantsTTL


class PartnerUserAdminInline(admin.TabularInline):
    model = PartnerUser
    exclude = ('passportuid',)  # No reason to show passportuid in the interface
    extra = 0


class PartnerAdminImageWidget(AdminFileWidget):
    def render(self, name, value, attrs=None):
        output = []

        if value and getattr(value, "url", None):
            output.append(
                u' <img src="%s?%s" alt="%s" />' % (value.url, os_time.time() * 1000, value)
            )

        output.append(super(AdminFileWidget, self).render(name, value, attrs))

        return mark_safe(u''.join(output))


class BillingOrderWidget(forms.TextInput):
    def render(self, name, value, *args, **kwargs):
        html = super(BillingOrderWidget, self).render(name, value, *args, **kwargs)
        if self.instance.billing_order_id:
            html += mark_safe(
                u'\n <a href="{}/campaign/{}">{}</a>'.format(
                    getattr(settings, 'PARTNERKA_URL', ''),
                    self.instance.billing_order_id,
                    _(u'Перейти в партнерский интерфейс')
                )
            )

        return html


class PartnerAdminForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(PartnerAdminForm, self).__init__(*args, **kwargs)
        if hasattr(self, 'instance') and 'billing_client_id' in self.fields:
            self.fields['billing_client_id'].widget.instance = self.instance

    def clean_query_module_name(self):
        allowed_chars = string.ascii_lowercase + string.digits + '_'
        query_module_name = self.cleaned_data['query_module_name'] or ''

        if any([char not in allowed_chars for char in query_module_name]):
            raise forms.ValidationError(
                u'В имени модуля разрешены только: "%s"' % allowed_chars
            )

        return query_module_name

    class Meta:
        model = Partner
        widgets = {
            'billing_client_id': BillingOrderWidget,
        }
        exclude = []


class DohopResidencesFilter(SimpleListFilter):
    title = _(u'Страны')
    parameter_name = 'index_infos__residences__contains'

    def lookups(self, request, model_admin):
        return (
            ('RU',  _(u'Россия')),
            ('UA',  _(u'Украина')),
            ('TR',  _(u'Турция')),
        )

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(dohop_residences__contains=self.value())

        else:
            return queryset


class DohopIndexInfoFilter(SimpleListFilter):
    title = _(u'Информация по вендору дохоп в индексе')
    parameter_name = 'index_infos'

    def lookups(self, request, model_admin):
        return [
            (obj['id'], self._get_filter_name(obj))
            for obj in self._lookups_queryset_for_dohopinfo().values(
                'id', 'index__code', 'vendor__code'
            ).all()
        ]

    def _lookups_queryset_for_dohopinfo(self):
        return DohopVendorIndexInfo.objects.select_related('vendor', 'index')

    def _get_filter_name(self, params):
        return '{index__code} {vendor__code}'.format(**params)

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(index_infos=self.value())
        else:
            return queryset


class DohopVendorAdminForm(forms.ModelForm):
    class Meta:
        model = DohopVendor
        exclude = []


def _validate_file_size(field_file):
    if len(field_file) > MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES:
        raise forms.ValidationError(
            _(u'Файл %s превышает допустимый лимит %.2f МБ')
            % (field_file, MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES / 1024. / 1024.)
        )


def _validate_svg(field_file):
    try:
        safe_parse_xml(field_file)
    except Exception:
        raise forms.ValidationError(_(u'%s: формат файла не поддерживается' % field_file))


class DohopIndexInline(admin.TabularInline):
    model = DohopVendorIndexInfo
    extra = 0


class BasePartnerAdmin(object):
    object_history_template = 'admin/order/base_partner/history.html'

    partner_fields = [
        (_(u'Партнёр'), {'fields': [
            'title',
            'site_url',
            'code',
            'review_percent',
            'review_percent_ru',
            'review_percent_kz',
            'review_percent_ua',
            'review_percent_com',
            'review_percent_tr',
            'is_aviacompany',
        ]}), ]

    additional_info_fields = []

    common_fields = [
        (_(u'Переводы в русской национальной версии'), {'fields': L_field.admin_fields(Partner, ['national_ru_title'])}),
        (_(u'Переводы в украинской национальной версии'), {'fields': L_field.admin_fields(Partner, ['national_ua_title'])}),
        (_(u'Переводы в казахской национальной версии'), {'fields': L_field.admin_fields(Partner, ['national_kz_title'])}),
        (_(u'Переводы в турецкой национальной версии'), {'fields': L_field.admin_fields(Partner, ['national_tr_title'])}),
        (_(u'Переводы в com версии'), {'fields': L_field.admin_fields(Partner, ['national_com_title'])}),
        (_(u'Авиабилеты: опрос партнёра'), {'fields': [
            ('enabled_in_ticket_ru', 'enabled_in_mobile_ticket_ru'),
            ('enabled_in_ticket_ua', 'enabled_in_mobile_ticket_ua'),
            ('enabled_in_ticket_kz', 'enabled_in_mobile_ticket_kz'),
            ('enabled_in_ticket_tr', 'enabled_in_mobile_ticket_tr'),
            ('enabled_in_ticket_com', 'enabled_in_mobile_ticket_com'),
            'foreign_currency',
        ]}),
        (_(u'Расписания: опрос партнёра'), {'fields': [
            ('enabled_in_rasp_ru', 'enabled_in_mobile_rasp_ru'),
            ('enabled_in_rasp_ua', 'enabled_in_mobile_rasp_ua'),
            ('enabled_in_rasp_kz', 'enabled_in_mobile_rasp_kz'),
            ('enabled_in_rasp_tr', 'enabled_in_mobile_rasp_tr'),
        ]}),
        (_(u'Колдунщик: показ предложений'), {'fields': [
            'enabled_in_wizard_ru',
            'enabled_in_wizard_ua',
            'enabled_in_wizard_kz',
            'enabled_in_wizard_tr',
        ]}),
        (
            _(u'Информация о партнёре в разделе "Партнёры"'),
             {'fields': [
                 ('logo2_svg_ru', 'logo2_svg2png_ru'),
                 ('logo2_svg_ua', 'logo2_svg2png_ua'),
                 ('logo2_svg_kz', 'logo2_svg2png_kz'),
                 ('logo2_svg_tr', 'logo2_svg2png_tr'),
                 ('logo2_svg_com', 'logo2_svg2png_com'),
                 'icon_svg',
                 ('logo_svg_ru', 'logo_svg2png_ru'),
                 ('logo_svg_ua', 'logo_svg2png_ua'),
                 ('logo_svg_kz', 'logo_svg2png_kz'),
                 ('logo_svg_tr', 'logo_svg2png_tr'),
                 ('logo_svg_com', 'logo_svg2png_com'),
             ]}
        ),
    ]
    # inlines = [PartnerEmailAdminInline, PartnerUserAdminInline, CPCPriceInline]
    list_filter = [
        'foreign_currency',
        'enabled_in_ticket_ru',
        'enabled_in_mobile_ticket_ru',
        'enabled_in_ticket_ua',
        'enabled_in_mobile_ticket_ua',
        'enabled_in_ticket_kz',
        'enabled_in_mobile_ticket_kz',
        'enabled_in_ticket_tr',
        'enabled_in_mobile_ticket_tr',
        'enabled_in_rasp_ru',
        'enabled_in_mobile_rasp_ru',
        'enabled_in_rasp_ua',
        'enabled_in_mobile_rasp_ua',
        'enabled_in_rasp_kz',
        'enabled_in_mobile_rasp_kz',
        'enabled_in_rasp_tr',
        'enabled_in_mobile_rasp_tr',
    ]
    list_display = (
        'title', 'code'
    )
    formfield_overrides = {
        CustomImageField: {'widget': PartnerAdminImageWidget},
        SvgImageField: {'widget': PartnerAdminImageWidget},
    }

    def get_fieldsets(self, request, obj=None):
        return self.partner_fields + self.additional_info_fields + self.common_fields

    def construct_change_message(self, request, form, formsets, add=False):
        if add:
            return 'created'
        new = form.instance
        old = form.instance.history.all()[1:2]
        if not old:
            return 'No previous version found'
        return self.diff_versions(new, old[0].instance)

    @staticmethod
    def diff_versions(new, old):
        changes = []
        for field in new._meta.fields:
            if hasattr(new, field.name) and \
                    hasattr(old, field.name):
                old_value = getattr(old, field.name, '')
                new_value = getattr(new, field.name)
                if old_value != new_value:
                    changes.append(u'{}({} -> {})'.format(field.name, old_value, new_value))

        return '\n'.join(changes)

    def history_view(self, request, object_id, extra_context=None):
        """The 'history' admin view for this model."""

        from django.contrib.admin.models import LogEntry

        # First check if the user can see this history.
        model = self.model
        obj = self.get_object(request, unquote(object_id))
        if obj is None:
            return self._get_obj_does_not_exist_redirect(request, model._meta, object_id)

        if not self.has_change_permission(request, obj):
            raise PermissionDenied

        # Then get the history for this object.
        opts = model._meta
        action_list = LogEntry.objects.filter(
            object_id=unquote(object_id),
            content_type=get_content_type_for_model(model)
        ).select_related().order_by('-action_time')

        partnerka_action_list = UpdateHistoryRecord.objects.filter(
            partner_id=unquote(object_id),
        ).select_related().order_by('-time')

        const_dt = datetime(2020, 1, 1)
        context = dict(
            self.admin_site.each_context(request),
            title=_('Change history: %s') % force_text(obj),
            # hack, because in python2 there is no key argument in heapq.merge
            # so we merge tuples in which first argument is a key
            action_list=map(
                operator.itemgetter(2),
                heapq.merge(
                    map(lambda al: (const_dt - al.action_time, 0, al), action_list),
                    map(lambda pal: (const_dt - pal.time, 1, pal), partnerka_action_list),
                )
            ),
            module_name=capfirst(force_text(opts.verbose_name_plural)),
            object=obj,
            opts=opts,
            preserved_filters=self.get_preserved_filters(request),
        )
        context.update(extra_context or {})

        request.current_app = self.admin_site.name

        return TemplateResponse(request, self.object_history_template, context)


class DohopVendorAdmin(BasePartnerAdmin, RaspExportModelAdmin):
    form = DohopVendorAdminForm

    additional_info_fields = BasePartnerAdmin.additional_info_fields + [(
        _(u'Dohop'),
        {
            'fields': [
                't_type',
                'dohop_cache_ttl',
                'view_query_module_name',
                'enabled',
                'dohop_code',
                'dohop_id'
            ]}
    ), ]

    common_fields = BasePartnerAdmin.common_fields + [
        (_(u'Описания'), {'fields': [
            'description_ru_ru',
            'description_ua_uk',
            'description_ua_ru',
            'description_tr_tr',
            'description_tr_en',
        ], 'classes': ('collapse',)}),
    ]

    list_filter = BasePartnerAdmin.list_filter + [
        'index_infos__index__code',
        DohopResidencesFilter,
        DohopIndexInfoFilter,
    ]

    ordering = ('dohop_code',)
    readonly_fields = (
        'dohop_residences', 'dohop_langs', 'dohop_vendor_info', 'dohop_code',
        'dohop_id', 'code', 'view_query_module_name'
    )

    list_display = ('title', 'dohop_code', 'dohop_id', 'enabled', 'show_indexes')

    inlines = (DohopIndexInline, DohopVariantsTTLAdminInline)

    def get_queryset(self, request):
        objects = super(DohopVendorAdmin, self).get_queryset(
            request
        ).prefetch_related(
            'index_infos__index'
        )
        return objects

    def view_query_module_name(self, obj):
        return obj.partner.query_module_name

    view_query_module_name.short_description = _(u'Версия модуля')

    def show_indexes(self, obj):
        return ','.join(i.index.code for i in obj.index_infos.all())
    show_indexes.short_description = _(u'Индексы')

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)
        urlpatterns = super(DohopVendorAdmin, self).get_urls()
        info = self.model._meta.app_label, self.model._meta.model_name
        urlpatterns = [
            url(r'logos/$', wrap(self.logos_view), name='%s_%s_logos' % info),
        ] + urlpatterns
        return urlpatterns

    @csrf_protect_m
    def logos_view(self, request, extra_context=None):
        """
        Show all_vendors_logos
        """

        opts = self.model._meta
        app_label = opts.app_label
        if not self.has_change_permission(request, None):
            raise PermissionDenied

        def vendor_info(v):
            nationals = 'ru ua kz tr com'.split()
            svgs = [getattr(v, 'logo_svg_{}'.format(national)) for national in nationals]
            pngs = [getattr(v, 'logo_svg2png_{}'.format(national)) for national in nationals]
            return {
                'id': v.id,
                'dohop_id': v.dohop_id,
                'code': v.code,
                'title': v.title,
                'svg_logos': svgs,
                'png_logos': pngs,
                'nationals': nationals,
            }

        vendors = [vendor_info(v) for v in self.get_queryset(request)]
        vendors = [v for v in vendors if any(v['png_logos']) or any(v['svg_logos'])]
        vendors.sort(key=lambda v: v['dohop_id'])
        context = dict(
            self.admin_site.each_context(request),
            module_name=force_text(opts.verbose_name_plural),
            title="DohopVendor's logos",
            media=self.media,
            opts=opts,
            vendors=vendors,
            salt=int(os_time.time() * 10**9),
        )
        request.current_app = self.admin_site.name
        context.update(extra_context or {})
        return TemplateResponse(
            request, 'admin/%s/%s/logos_view.html' % (app_label, opts.model_name),
            context
        )


admin.site.register(DohopVendor, DohopVendorAdmin)
admin.site.register(DohopIndex)


class IsDeletedFilter(admin.SimpleListFilter):
    title = _(u'Удален')
    parameter_name = 'is_deleted'

    def lookups(self, request, model_admin):
        return [('all', 'All'), ('yes', 'Да'), ('no', 'Нет')]

    def queryset(self, request, queryset):
        if self.value() == 'yes':
            return queryset.filter(is_deleted=True)
        elif self.value() == 'no':
            return queryset.filter(is_deleted=False)
        else:
            return queryset

    def value(self):
        """
        Make "No" default instead of "All"
        """
        value = super(IsDeletedFilter, self).value()
        if value is None:
            value = 'no'
        return str(value)

    def choices(self, changelist):
        """
        By default, value of "All" options is None. It does not suit our implementation of default value, so we
        override this method to get rid of default "All" option.
        """
        for lookup, title in self.lookup_choices:
            yield {
                'selected': self.value() == str(lookup),
                'query_string': changelist.get_query_string({self.parameter_name: lookup}),
                'display': title,
            }


class PartnerAdmin(BasePartnerAdmin, RaspExportModelAdmin):
    def regionalizepartnerqueryrule_url(self, obj):
        rule_count = RegionalizePartnerQueryRule.objects.filter(partner=obj).count()

        if rule_count:
            count_text = xngettext(
                rule_count,
                u'<n/> правило',
                u'<n/> правила',
                u'<n/> правил',
            )

            return '<a href="/admin/order/regionalizepartnerqueryrule/?partner__id__exact=%s">%s %s</a>' % (
                obj.id, count_text[0], count_text[1]
            )

        else:
            return u'-'

    regionalizepartnerqueryrule_url.allow_tags = True
    regionalizepartnerqueryrule_url.short_description = "Рег. правила"

    def check_partner_answer_url(self, obj):
        return u'<a href="/admin/order/partner/{}/change/track">{}</a>'.format(obj.id, u'Опросить')

    check_partner_answer_url.allow_tags = True
    check_partner_answer_url.short_description = u'Опросить'

    form = PartnerAdminForm

    additional_info_fields = BasePartnerAdmin.additional_info_fields + [
        (
            _(u'Партнёр дополнительно'),
            {'fields': [
                't_type',
                'variant_cache_ttl',
                'billing_client_id',
                'query_module_name',
                'marker',
                'can_fetch_by_daemon',
                'can_sale_by_installments',
                'disabled',
                'enabled_with_negative_balance',
                'is_deleted',
            ]}
        ),
    ]

    common_fields = BasePartnerAdmin.common_fields + [
        (
            _(u'Биллинг'),
            {'fields': [
                'current_balance',
                'notify_balance_threshold',
                'low_balance_notification_sent',
                'null_balance_notification_sent',
                'statistic_reminder_sent',
                'balance_updated_at',
                'pricing_model',
                'use_in_update_conversions',
            ]}
        ),
        (
            _(u'_Биллинг'), {'fields': [
                'billing_order_id',
                'click_price',
                'click_price_ru',
                'click_price_ua',
                'click_price_kz',
                'click_price_tr',
                'click_price_com',
                'billing_datasource_id_dev',
                'billing_datasource_id_testing',
                'billing_datasource_id_production',
            ], 'classes': ('collapse',)}
        ),
    ]

    list_filter = BasePartnerAdmin.list_filter + [
        'pricing_model', 'use_in_update_conversions', 'can_sale_by_installments', IsDeletedFilter,
    ]

    inlines = [PartnerEmailAdminInline, PartnerUserAdminInline, CPCPriceInline, PartnerVariantsTTLAdminInline]

    list_display = (
        'title',
        'code',
        'disabled',
        't_type',
        'regionalizepartnerqueryrule_url',
        'check_partner_answer_url',
    )
    readonly_fields = (
        'low_balance_notification_sent', 'null_balance_notification_sent', 'statistic_reminder_sent',
    )

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(PartnerAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/change/track/$', wrap(self.track_view), name='%s_%s_track' % info),
            url(r'^(.+)/change/track/result/$', wrap(self.track_result_view), name='%s_%s_track_result' % info),
        ] + urlpatterns

        return urlpatterns

    @jsonp_response_m
    def track_result_view(self, request, key):
        """ Show import results """

        r = requests.get(
            urljoin(settings.TICKET_DAEMON_API_URL, 'jsendapi/v3/track/result/'),
            params={'trid': request.GET.get('trid')},
            timeout=60,
            verify=False
        )

        if not r.ok:
            raise Exception('%r %r %r' % (r.status_code, r.url, r.content))

        return r.json()['data']

    def track_start(self, data, partner):
        query_params = dict([
            ('point_from', data['point_from'].point_key),
            ('point_to', data['point_to'].point_key),
            ('date_forward', data['when'].strftime('%Y-%m-%d')),
            ('date_backward', data['date_backward'] and
             data['date_backward'].strftime('%Y-%m-%d')),
            ('klass', data['klass']),
            ('passengers', {
                k: data[k]
                for k in ('adults', 'children', 'infants')
            }),
            ('national_version', data['national_version']),
            ('service', settings.PROJECT_CODE),
            ('lang', data['lang'])
        ])

        r = requests.post(
            urljoin(settings.TICKET_DAEMON_API_URL, 'jsendapi/v3/track/start/'),
            headers={
                'Content-type': 'application/json',
            },
            data=json.dumps({
                'query_params': query_params,
                'p_code': partner.code,
            }),
            timeout=60,
            verify=False
        )

        if not r.ok:
            raise Exception('%r %r %r' % (r.status_code, r.url, r.content))

        return r.json()['data']['trid']

    @csrf_protect_m
    def track_view(self, request, partner_id):
        """ Query partner interface """

        model = self.model
        opts = model._meta
        app_label = opts.app_label

        partner = self.get_object(request, unquote(partner_id))

        context = {}

        if request.method == 'POST':
            form = TrackDaemonImportForm(request.POST)

            if form.is_valid():
                params = {
                    form[field_name].html_name:
                    field.prepare_value(form.cleaned_data[field_name])
                    for field_name, field in form.fields.items()
                }

                try:
                    params['trid'] = self.track_start(
                        form.cleaned_data, partner)

                except Exception:
                    context.setdefault('errors', []). \
                        append('track_start error: %s' % format_exc())

                else:
                    return HttpResponseRedirect('?' + urlencode(
                        filter(lambda (k, v): v is not None, params.items())
                    ))

        else:
            form = TrackDaemonImportForm(initial=request.GET)

            trid = request.GET.get('trid')

            if trid:
                context.update({
                    'trid': trid,
                })

        context.update({
            'title': _(u'Проверить ответ партнера по направлению'),
            'form': form,
            'media': self.media + form.media,
            'object_id': partner_id,
            'original': partner,
            'is_popup': "_popup" in request,
            'app_label': app_label,
            'has_change_permission': self.has_change_permission(request, partner),
            'opts': opts,
            'content_type_id': ContentType.objects.get_for_model(self.model).id,
        })

        return TemplateResponse(
            request,
            "admin/%s/%s/track.html" % (app_label, opts.object_name.lower()),
            context,
        )

    def get_readonly_fields(self, request, obj=None):
        if obj:
            if obj.is_deleted:
                all_fields = flatten_fieldsets(self.get_fieldsets(request, obj))
                all_fields.remove('is_deleted')
                return all_fields
            if not obj.disabled:
                return self.readonly_fields + ('is_deleted',)
        return self.readonly_fields

    def has_delete_permission(self, request, obj=None):
        if not obj or not obj.is_deleted:
            return True
        messages.add_message(request, messages.WARNING, 'Партнер считается удаленным, редактирование недоступно')
        return False


class ForeignKeyRawModelIdWidget(ForeignKeyRawIdWidget):
    def __init__(self, model, lookup_params=None, *args, **kwargs):
        self.lookup_params = lookup_params

        super(ForeignKeyRawModelIdWidget, self).__init__(
            ManyToOneRel(None, model, 'id'), admin.site, *args, **kwargs
        )

    def url_parameters(self):
        params = super(ForeignKeyRawModelIdWidget, self).url_parameters()

        if self.lookup_params:
            params.update(self.lookup_params)

        return params


class ModelRawIdChoiceField(forms.ModelChoiceField):
    def __init__(self, model, *args, **kwargs):
        lookup_params = kwargs.pop('lookup_params', None)

        super(ModelRawIdChoiceField, self).__init__(
            model.objects,
            widget=ForeignKeyRawModelIdWidget(model, lookup_params),
            *args, **kwargs
        )


KLASS_CHOICES = [
    ('economy', 'Эконом'),
    ('business', 'Бизнес'),
]


NATIONAL_VERSION_CHOICES = [
    (v, v)
    for v in settings.AVIA_NATIONAL_VERSIONS
]

LANG_CHOICES = [
    (v, v)
    for v in settings.FORM_LANGUAGES
]


class TrackDaemonImportForm(forms.Form):
    from_code = forms.CharField(required=False)
    from_station = ModelRawIdChoiceField(
        Station,  required=False, lookup_params={'t_type__code': 'plane'})
    from_settlement = ModelRawIdChoiceField(Settlement, required=False)

    to_code = forms.CharField(required=False)
    to_station = ModelRawIdChoiceField(
        Station, required=False, lookup_params={'t_type__code': 'plane'})
    to_settlement = ModelRawIdChoiceField(Settlement, required=False)

    when = forms.DateField(required=True, widget=AdminDateWidget)
    date_backward = forms.DateField(required=False, widget=AdminDateWidget)

    adults = forms.IntegerField(required=False, initial=1)
    children = forms.IntegerField(required=False, initial=0)
    infants = forms.IntegerField(required=False, initial=0)

    klass = forms.ChoiceField(
        required=False, choices=KLASS_CHOICES, initial=KLASS_CHOICES[0])

    national_version = forms.ChoiceField(
        required=False, choices=NATIONAL_VERSION_CHOICES,
        initial=NATIONAL_VERSION_CHOICES[0])

    lang = forms.ChoiceField(
        required=False, choices=LANG_CHOICES,
        initial=LANG_CHOICES[0]
    )

    def clean_from_code(self):
        self._point_from_by_code = _point_by_code(
            self.cleaned_data['from_code'])

        if self._point_from_by_code:
            return self.cleaned_data['from_code']

    def clean_to_code(self):
        self._point_to_by_code = _point_by_code(self.cleaned_data['to_code'])

        if self._point_to_by_code:
            return self.cleaned_data['to_code']

    def clean_adults(self):
        val = self.cleaned_data['adults']
        return val if val is not None else self.fields['adults'].initial

    def clean_children(self):
        val = self.cleaned_data['children']
        return val if val is not None else self.fields['children'].initial

    def clean_infants(self):
        val = self.cleaned_data['infants']
        return val if val is not None else self.fields['infants'].initial

    def clean(self):
        data = self.cleaned_data

        point_from = self._point_from_by_code

        if point_from:
            data['from_station'] = None
            data['from_settlement'] = None

        elif data.get('from_station'):
            data['from_settlement'] = None

        point_from = point_from or \
            data.get('from_station') or data.get('from_settlement')

        if not point_from:
            raise forms.ValidationError(_(u'Нужно указать пункт отправления'))

        data['point_from'] = point_from

        point_to = self._point_to_by_code

        if point_to:
            data['to_station'] = None
            data['to_settlement'] = None

        elif data.get('to_station'):
            data['to_settlement'] = None

        point_to = point_to or \
            data.get('to_station') or data.get('to_settlement')

        if not point_to:
            raise forms.ValidationError(_(u'Нужно указать пункт прибытия'))

        data['point_to'] = point_to

        if data['point_from'] == data['point_to']:
            raise forms.ValidationError(
                _(u'Пукнты отправления и прибытия должны различаться'))

        return self.cleaned_data


admin.site.register(Partner, PartnerAdmin)

admin.site.register(CoachService)


class CustomPartnersFilter(SimpleListFilter):
    title = _(u'Партнеры')
    parameter_name = 'partner__id__exact'

    def lookups(self, request, model_admin):
        partner_ids = RegionalizePartnerQueryRule.objects.values_list("partner", flat=True).distinct()
        partners = Partner.objects.in_bulk(partner_ids).items()

        return partners

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(partner=self.value())

        else:
            return queryset


class RegionalizePartnerQueryForm(forms.ModelForm):
    def clean_iatas(self):
        iatas = filter(
            None,
            [iata.strip().upper() for iata in re.split('[\r\n;]+', self.cleaned_data['iatas'])]
        )

        unknown_iatas = set()
        settlements = set()

        if iatas:
            for iata in iatas:
                settlement = None

                try:
                    settlement = Settlement.objects.get(iata=iata)

                except Settlement.DoesNotExist:
                    pass

                if not settlement:
                    try:
                        station = Station.get_by_code('iata', iata)
                        settlement = station.settlement

                    except Station.DoesNotExist:
                        unknown_iatas.add(iata)
                        continue

                settlements.add(settlement)

            if unknown_iatas:
                raise forms.ValidationError(
                    gettext(u'Неизвестные коды ИАТА') + u': ' + u'; '.join(unknown_iatas)
                )

            else:
                self.multiple_settlements_to = settlements

        return ';'.join(iatas)

    iatas = forms.CharField(
        label=u'IATA\'s',
        widget=forms.Textarea(
            attrs={
                'cols': 120,
                'rows': 5
            }
        ),
        required=False,
        help_text=_(u'Если заполнен список IATA, то блок "To" полностью игнорируется. Пример: "SVX;SVO;DME".')
    )

    class Meta:
        model = RegionalizePartnerQueryRule
        exclude = []


class RegionalizePartnerQueryRuleAdmin(RaspExportModelAdmin):
    form = RegionalizePartnerQueryForm

    list_display = ('__unicode__', 'display_from', 'display_to',
                    'display_user_point', 'exclude')

    search_fields = ['settlement_from__title']

    raw_id_fields = (
        "settlement_from", "country_from", "pseudo_region_from",
        "settlement_to", "country_to", "pseudo_region_to",
        "user_settlement", "user_country", "user_pseudo_region",
    )

    fieldset_header = ('', {'fields': ("partner", "exclude")})
    fieldset_from = ('From', {'fields': ("settlement_from", "country_from", "pseudo_region_from")})
    fieldset_to = ('To', {'fields': ("settlement_to", "country_to", "pseudo_region_to")})
    fieldset_multiple_to = ('Multiple To', {'fields': ("iatas",)})
    fieldset_user = ('User', {'fields': ("user_settlement", "user_country", "user_pseudo_region")})
    fieldset_schedule = ('Schedule', {'fields': {'start_date', 'end_date', 'week_days'}})

    list_filter = [CustomPartnersFilter]

    def get_fieldsets(self, request, obj=None):
        if obj:
            fieldsets = (
                self.fieldset_header,
                self.fieldset_from,
                self.fieldset_to,
                self.fieldset_user,
                self.fieldset_schedule,
            )
        else:
            fieldsets = (
                self.fieldset_header,
                self.fieldset_from,
                self.fieldset_to,
                self.fieldset_multiple_to,
                self.fieldset_user,
                self.fieldset_schedule,
            )

        return fieldsets

    def response_add(self, request, obj):
        if not obj._get_pk_val():
            request.POST = request.POST.copy()
            request.POST.pop('_continue', None)

        return super(RegionalizePartnerQueryRuleAdmin, self).response_add(request, obj)

    @transaction.atomic
    def save_model(self, request, obj, form, change, *args, **kwargs):
        multiple_settlements_to = getattr(form, 'multiple_settlements_to', None)

        if multiple_settlements_to:
            for settlement in multiple_settlements_to:
                RegionalizePartnerQueryRule.objects.get_or_create(
                    partner=obj.partner,
                    settlement_from=obj.settlement_from,
                    country_from=obj.country_from,
                    settlement_to=settlement,
                    defaults={
                        'pseudo_region_from': obj.pseudo_region_from,
                        'user_settlement': obj.user_settlement,
                        'user_country': obj.user_country,
                        'user_pseudo_region': obj.user_pseudo_region,
                        'exclude': obj.exclude,
                    }
                )

        else:
            return super(RegionalizePartnerQueryRuleAdmin, self).save_model(
                request, obj, form, change, *args, **kwargs
            )

        return True

    def display_from(self, rpq):
        point = rpq.get_point_from()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_from.short_description = _(u"Откуда")

    def display_to(self, rpq):
        point = rpq.get_point_to()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_to.short_description = _(u"Куда")

    def display_user_point(self, rpq):
        point = rpq.get_user_point()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_user_point.short_description = _(u"Положение пользователя")


admin.site.register(RegionalizePartnerQueryRule, RegionalizePartnerQueryRuleAdmin)


class QueryBlackListForm(forms.ModelForm):
    active_to = forms.DateField(
        initial=datetime.today() + timedelta(days=3),
        widget=AdminDateWidget
    )

    def clean(self):
        data = self.cleaned_data

        price_from = data.get("price_from", None)
        price_to = data.get("price_to", None)
        currency = data.get("currency", None)

        settlement_from = data.get("settlement_from", None)
        station_from = data.get("station_from", None)
        country_from = data.get("country_from", None)

        settlement_to = data.get("settlement_to", None)
        station_to = data.get("station_to", None)
        country_to = data.get("country_to", None)

        if (price_from or price_to) and not currency:
            raise forms.ValidationError(_(u"Пожалуйста, выберите валюту."))

        if currency and not (price_from or price_to):
            raise forms.ValidationError(_(u"Пожалуйста, укажите цену от и/или до."))

        flight_number = data.get('flight_number', None)
        klass = data.get('klass', None)
        at_least_one_field = settlement_from or station_from or country_from or settlement_to or station_to or country_to
        at_least_one_field = at_least_one_field or price_from or price_to or flight_number or klass

        if not at_least_one_field:
            raise forms.ValidationError(_(u'Заполните хотя бы одно гео-поле, поле цены, номера рейса или класса обслуживания.'))

        return data

    class Meta:
        model = QueryBlackList
        exclude = []


class QueryBlackListAdmin(RaspExportModelAdmin):
    form = QueryBlackListForm

    def formfield_for_dbfield(self, db_field, **kwargs):
        formfield = super(QueryBlackListAdmin, self).formfield_for_dbfield(db_field, **kwargs)

        if db_field.name == 'description':
            formfield.widget = forms.Textarea(attrs=formfield.widget.attrs)

        return formfield

    def display_from(self, qbl):
        froms = [f.L_popular_title() for f in filter(None, [qbl.country_from, qbl.settlement_from, qbl.station_from])]

        return ', '.join(froms)

    display_from.short_description = _(u"Откуда")

    def display_to(self, qbl):
        froms = [f.L_title() for f in filter(None, [qbl.country_to, qbl.settlement_to, qbl.station_to])]

        return ', '.join(froms)

    display_to.short_description = _(u"Куда")

    def display_full_flight_number(self, qbl):
        company_iata = None

        if getattr(qbl, "company", None):
            company_iata = qbl.company.iata

        return '-'.join(filter(None, [company_iata, qbl.flight_number]))

    display_full_flight_number.short_description = _(u"АК/Рейс")

    def display_partner(self, qbl):
        return qbl.partner.L_title() if qbl.partner else None

    display_partner.short_description = _(u"Партнер")

    list_display = (
        "__unicode__",
        "display_partner",
        "display_from",
        "display_to",
        "display_full_flight_number",
        "when_from", "when_to",
        "price_from", "price_to", "currency",
        "active_to", "national_version", "active", "allow"
    )

    raw_id_fields = (
        "settlement_from", "station_from", "country_from",
        "settlement_to", "station_to", "country_to",
        "company"
    )

admin.site.register(QueryBlackList, QueryBlackListAdmin)


def _point_by_code(code):
    try:
        return Point.get_by_key(code)
    except (ValueError, ObjectDoesNotExist):
        pass

    code = code.lower()

    try:
        return Settlement.objects.get(iata__iexact=code)
    except Settlement.DoesNotExist:
        pass

    try:
        return Settlement.objects.get(sirena_id__iexact=code)
    except Settlement.DoesNotExist:
        pass

    try:
        return Station.get_by_code('iata', code)
    except Station.DoesNotExist:
        pass

    try:
        return Station.get_by_code('sirena', code)
    except Station.DoesNotExist:
        pass


class EmailTypeAdmin(RaspExportModelAdmin):
    list_display = ('code', 'title')

admin.site.register(EmailType, EmailTypeAdmin)


class PartnerUserAdmin(RaspExportModelAdmin):
    list_display = ('role', 'login', 'name', 'partner')
    exclude = ('passportuid',)  # No reason to show passportuid in the interface

admin.site.register(PartnerUser, PartnerUserAdmin)


class BlablacarTokenAdmin(RaspExportModelAdmin):
    list_display = ('token_title', 'expire_msk_dt', 'receive_msk_dt')
    readonly_fields = ('token', 'expire_msk_dt', 'receive_msk_dt')

    ordering = ('-expire_msk_dt', )

    actions = None

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def token_title(self, blablacar_token):
        return blablacar_token.token[:8]


admin.site.register(BlablacarToken, BlablacarTokenAdmin)


class DefaultClickPriceAdmin(ModelAdmin):
    list_display = ('national_version', 'billing_price')

admin.site.register(DefaultClickPrice, DefaultClickPriceAdmin)


class PriceListAdmin(ModelAdmin):
    list_display = ('settlement_from', 'settlement_to', 'month', 'is_one_way', 'national_version', 'price')
    list_filter = (
        'is_one_way',
        'national_version',
    )
    search_fields = ('settlement_from__title',)
    readonly_fields = ('settlement_from', 'settlement_to', 'month', 'is_one_way', 'national_version', 'price')

admin.site.register(PriceList, PriceListAdmin)


def _partner_name(partner):
    return _(u'{} [{}]'.format(partner.L_title(), partner.code))


class PartnerFilter(admin.SimpleListFilter):
    title = _(u'Партнер')
    parameter_name = 'partner'

    def lookups(self, request, model_admin):
        return sorted(
            (
                (partner.id, _partner_name(partner))
                for partner in Partner.objects.filter(t_type=TransportType.PLANE_ID).all()
            ),
            key=lambda x: x[1],
        )

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(partner=self.value())

        else:
            return queryset


class CPCPriceAdmin(ModelAdmin):
    def display_partner(self, qbl):
        return _partner_name(qbl.partner) if qbl.partner else None

    display_partner.short_description = _(u'Партнер')

    list_display = (
        'display_partner',
        'national_version',
        'n_redirects',
        'cpa_sum',
        'eCPC',
        'eCPC_rasp_direct',
        'eCPC_sovetnik_direct',
        'eCPC_wizard_direct',
        'eCPC_wizard_indirect',
    )
    list_filter = (
        PartnerFilter,
        'national_version',
    )
    search_fields = ('partner__code', )

admin.site.register(CPCPrice, CPCPriceAdmin)


class RedirectTypeAdmin(ModelAdmin):
    list_display = ('code', 'name')

admin.site.register(RedirectType, RedirectTypeAdmin)


class ConversionByRedirectTypeAdmin(ModelAdmin):
    def display_redirect_type(self, qbl):
        return qbl.redirect_type.name if qbl.redirect_type else None

    display_redirect_type.short_description = _(u'Тип редиректа')
    list_display = ('national_version', 'display_redirect_type', 'conversion', 'updated_at')


admin.site.register(ConversionByRedirectType, ConversionByRedirectTypeAdmin)


class ClickPriceMultiplierByRedirectTypeAdmin(ModelAdmin):
    def display_redirect_type(self, qbl):
        return qbl.redirect_type.name if qbl.redirect_type else None

    display_redirect_type.short_description = _(u'Тип редиректа')
    list_display = ('national_version', 'display_redirect_type', 'multiplier', 'updated_at')


admin.site.register(ClickPriceMultiplierByRedirectType, ClickPriceMultiplierByRedirectTypeAdmin)


class AmadeusMerchantAdmin(BasePartnerAdmin, RaspExportModelAdmin):

    additional_info_fields = BasePartnerAdmin.additional_info_fields + [
        (_(u'Amadeus'), {'fields': [
            'merchant_id',
            't_type',
            'variant_cache_ttl',
            'enabled',
        ]})
    ]

    common_fields = BasePartnerAdmin.common_fields + [
        (_(u'Параметры запроса'), {'fields': [
            ('currency_ru', 'language_ru'),
            ('currency_com', 'language_com'),
            ('currency_kz', 'language_kz'),
            ('currency_ua', 'language_ua'),
            ('currency_tr', 'language_tr'),
        ]}),
        (_(u'Описания'), {'fields': [
            'description_ru_ru',
            'description_ua_uk',
            'description_ua_ru',
            'description_tr_tr',
            'description_tr_en',
            'description_kz_ru',
        ], 'classes': ('collapse',)}),
    ]

    ordering = ('code',)
    readonly_fields = (
        'code', 't_type',
    )
    list_display = ('title', 'code', 'merchant_id', 'enabled')
    inlines = [AmadeusMerchantVariantsTTLAdminInline]


admin.site.register(AmadeusMerchant, AmadeusMerchantAdmin)


class UpdateHistoryRecordAdmin(admin.ModelAdmin):
    list_display = ('partner', 'time', 'action', 'description', 'updater_role', 'updater_yandex_login')
    list_filter = ('partner',)
    ordering = ('time',)
    readonly_fields = [field.name for field in UpdateHistoryRecord._meta.get_fields()]

    def has_add_permission(self, request, obj=None):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def get_actions(self, request):
        actions = super(UpdateHistoryRecordAdmin, self).get_actions(request)
        return list(action for action in actions if action != 'delete_selected')


admin.site.register(UpdateHistoryRecord, UpdateHistoryRecordAdmin)
