# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import os
from itertools import groupby
from operator import itemgetter

from logging import getLogger

from django.db import migrations


log = getLogger(__name__)


def create_users(users, partner_model, partner_user_model):
    for user in users:
        try:
            partner = partner_model.objects.get(id=user['partner']) if user['partner'] else None
        except Exception as exc:
            log.error("Can't get partner with id %r: %r", user['partner'], exc)

        try:
            partner_user_model.objects.create(
                role=user['role'],
                login=user['login'],
                name=user['name'],
                partner=partner
            )
        except Exception as exc:
            raise Exception("Couldn't create a partner_user: {}".format(exc))


def check_duplicate_owners(users):
    partner_owners = [u for u in users if u['role'] == 'owner' and u['partner']]

    partner_owners.sort(key=itemgetter('partner'))
    for partner_id, group in groupby(partner_owners, itemgetter('partner')):
        owners_of_partner = list(group)

        if len(owners_of_partner) > 1:
            raise Exception('Duplicate owners of the partner with id={}'.format(partner_id))


def fill_partneruser(apps, schema_editor):
    partner_model = apps.get_model("order", "Partner")
    partner_user_model = apps.get_model("order", "PartnerUser")

    with open(os.path.join(os.path.dirname(__file__), 'data/users.json')) as users_file:
        users = json.load(users_file)

        check_duplicate_owners(users)
        create_users(users, partner_model, partner_user_model)


def clear_partneruser(apps, schema_editor):
    partner_user = apps.get_model("order", "PartnerUser")
    partner_user.objects.all().delete()


class Migration(migrations.Migration):
    dependencies = [
        ('order', '0020_auto_20160920_0917'),
    ]

    operations = [
        migrations.RunPython(fill_partneruser, clear_partneruser, True),
    ]
