from logging import getLogger

from django.db import transaction
from django.db.models import F
from django.db.models.aggregates import Max
from django.http.response import HttpResponse

from travel.avia.library.python.common.models.partner import Partner, PartnerStatusUpdate
from travel.avia.library.python.common.utils.httpresponses import jsonp_response

log = getLogger(__name__)


class PermissionError(Exception):
    pass


@jsonp_response
def enable_partner(request):
    try:
        return _update_partner_status(
            partner_id=request.POST['partner_id'],
            yandex_login=request.POST['user_yandex_login'],
            role=request.POST['user_role'],
            disabled=False,
        )
    except PermissionError:
        return HttpResponse(status=401)


@jsonp_response
def disable_partner(request):
    try:
        return _update_partner_status(
            partner_id=request.POST['partner_id'],
            yandex_login=request.POST['user_yandex_login'],
            role=request.POST['user_role'],
            disabled=True,
        )
    except PermissionError:
        return HttpResponse(status=401)


@jsonp_response
def partner_statuses(request):
    partners = list(Partner.objects.all())

    last_status_updates = {
        status_update.partner_id: status_update
        for status_update in
            PartnerStatusUpdate.objects
            .annotate(max_time=Max('partner__status_updates__time'))
            .filter(time=F('max_time'))
    }

    return [
        {
            'id': partner.id,
            'code': partner.code,
            'disabled': partner.disabled,
            'title': partner.L_national_ru_title(),
            'campaign_id': partner.billing_datasource_id,
            'updated': {
                'at': status_update.time.strftime('%Y-%m-%dT%H:%M:%S'),
                'by': {
                    'yandex_login': status_update.updater_yandex_login,
                    'role': status_update.updater_role,
                }
            } if status_update else None
        }
        for partner, status_update in zip(
            partners,
            [last_status_updates.get(partner.id) for partner in partners]
        )
    ]


def _update_partner_status(partner_id, yandex_login, role, disabled):
    if role not in {'owner', 'manager'}:
        log.error(
            "The role %r can't update partner %r status", role, partner_id
        )
        raise PermissionError

    try:
        with transaction.atomic():
            partner = Partner.objects.get(pk=partner_id)
            partner.disabled = disabled
            partner.save()

            PartnerStatusUpdate(
                partner=partner,
                disabled=disabled,
                updater_yandex_login=yandex_login,
                updater_role=role,
            ).save()
    except Exception as exc:
        log.critical(
            "Couldn't update the partner's %r status: %r",
            partner_id, exc, exc_info=True
        )
        return {'success': False}

    return {'success': True}
