# -*- coding: utf-8 -*-

from __future__ import absolute_import

import os
import shutil
import logging
import sqlite3
import struct
from datetime import datetime, time
from itertools import repeat
from time import mktime

from django.db import connection
from django.conf import settings


DATA_ROOT = os.path.join(settings.DATA_PATH, 'wizard')
TMP_DIR = os.path.join(settings.DATA_PATH, 'tmp')
CACHE_ROOT = os.path.join(settings.CACHE_PATH, 'wizard')


log = logging.getLogger('precalc')


def msk_time(d):
    return d and int(mktime(d.timetuple()))


def int_time(t):
    return t.hour * 3600 + t.minute * 60 + t.second if t is not None else None


def time_int(t):
    minutes, seconds = divmod(t, 60)
    hours, minutes = divmod(minutes, 60)

    return time(hours, minutes, seconds)


def time_msk(t):
    return t and datetime.fromtimestamp(t)


def pack_run_mask(year_days):
    months = [int(year_days[m * 31:m * 31 + 31][::-1], 2) for m in xrange(12)]

    run_mask = struct.pack("!12I", *months)

    return run_mask


def unpack_run_mask(mask):
    months = struct.unpack("!12I", mask)

    return "".join(("1" if month & (1 << i) else "0") for month in months for i in range(31))


def format_conditions(conditions):
    expressions = []
    params = []

    for expression, param in conditions.items():
        if isinstance(param, (list, tuple)):
            expressions.append(
                expression.replace('?', '(%s)' % ','.join(repeat('?', len(param))))
            )
            params.extend(param)
        elif param is None and '?' not in expression:
            expressions.append(expression)
        else:
            expressions.append(expression)
            params.append(param)

    return ' AND '.join(expressions), params


def execute_with_conditions(conn, query, conditions):
    filter_clause, params = format_conditions(conditions)

    return conn.execute(query.format(filter_clause), params)


def get_db_path(db, cached=False):
    subpath = os.path.join(connection.get_db_name(), '%s.sqlite' % db)

    if cached:
        return os.path.join(CACHE_ROOT, subpath)

    return os.path.join(DATA_ROOT, subpath)


def open_db(db):
    path = get_db_path(db, cached=True)

    if not os.path.exists(path):
        path = get_db_path(db)

    conn = sqlite3.connect(path)

    return conn


class TmpDB(object):
    def __init__(self):
        db_name = 'schedule.tmp'

        self.db_path = os.path.join(TMP_DIR, '%s.sqlite' % db_name)

    def connect(self):
        conn = sqlite3.connect(self.db_path, 3600)

        conn.executescript("""
            PRAGMA synchronous = OFF;
            PRAGMA automatic_index = OFF;
            PRAGMA cache_size = 32768;
            PRAGMA ignore_check_constraints = ON;
            PRAGMA journal_mode = OFF;
            PRAGMA page_size = 65536;
        """)

        return conn

    def cleanup(self):
        if os.path.exists(self.db_path):
            os.remove(self.db_path)


class SQLiteSnapshot(object):
    def __init__(self, db_name, new=False, hack=False):
        self.db_path = get_db_path(db_name)
        self.db_path_tmp = os.path.join(TMP_DIR, '%s.sqlite' % db_name)
        self.hack = hack

        if hack:
            log.info("Hacking %s" % db_name)
            self.db_path_tmp = self.db_path

        elif new:
            log.info("Generating %s" % db_name)

            if os.path.exists(self.db_path_tmp):
                os.remove(self.db_path_tmp)

        else:
            log.info("Updating %s" % db_name)

            shutil.copy(self.db_path, self.db_path_tmp)

    def connect(self):
        conn = sqlite3.connect(self.db_path_tmp, 3600)

        conn.executescript("""
            PRAGMA synchronous = OFF;
            PRAGMA automatic_index = OFF;
            PRAGMA cache_size = 32768;
            PRAGMA ignore_check_constraints = ON;
            PRAGMA journal_mode = OFF;
            PRAGMA page_size = 65536;
        """)

        return conn

    def commit(self):
        if self.hack:
            log.info("Done.")
            return

        conn = self.connect()

        conn.executescript("""
           ANALYZE;
        """)

        conn.close()

        db_dir = os.path.dirname(self.db_path)

        if not os.path.exists(db_dir):
            os.makedirs(db_dir)

        os.rename(self.db_path_tmp, self.db_path)

        log.info("Done.")
