# -*- coding: utf-8 -*-

from django.db import models
from django.utils.translation import ugettext_lazy as _


class PromoPage(models.Model):
    """
    RASP-13589 Промо-страница
    """

    title = models.CharField(verbose_name=_(u'Название'), max_length=255)
    slogan = models.CharField(verbose_name=_(u'Слоган'), max_length=255)
    description = models.CharField(verbose_name=_(u'Описание страницы'), help_text=_(u'Мета-тег «description» (описание страницы)'), max_length=255)
    metric_id = models.CharField(verbose_name=_(u'id счетчика метрики'), max_length=100, null=True, blank=True)
    active = models.BooleanField(verbose_name=_(u'активная'))

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'промо-страница')
        verbose_name_plural = _(u'промо-страницы')
        ordering = ('title',)
        app_label = 'promosearch'


class PromoSettlement(models.Model):
    """
    RASP-13589 Город для отображения на промо-странице
    """

    DIRECTION_FROM = 1
    DIRECTION_TO = 2

    promo_page = models.ForeignKey('promosearch.PromoPage', verbose_name=_(u'Промо-страница'))
    settlement = models.ForeignKey('www.Settlement', verbose_name=_(u'Город'))
    direction = models.IntegerField(verbose_name=_(u'откуда/куда'),
                                    choices=[(DIRECTION_FROM, _(u'откуда')), (DIRECTION_TO, _(u'куда'))])
    priority = models.IntegerField(verbose_name=_(u'Приоритет'))

    def __unicode__(self):
        return self.settlement.L_title()

    @property
    def L_title(self):
        return self.settlement.L_title

    class Meta:
        verbose_name = _(u'промо-город')
        verbose_name_plural = _(u'промо-города')
        ordering = ('promo_page', 'priority', 'settlement')
        app_label = 'promosearch'


class ScheduleTable(models.Model):
    """
    RASP-13600 Таблица соответствия "поставщик+направление" -> "произвольный текст". Идея в том, чтобы вместо
    огромного кол-ва времен оправления мы выводили лаконичную строку: "раз в 15 минут" или "каждый час"
    """

    promo_page = models.ForeignKey('promosearch.PromoPage', verbose_name=_(u'Промо-страница'))
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), default=3)
    company = models.ForeignKey('www.Company', verbose_name=_(u'Поставщик'))
    station_from = models.ForeignKey('www.Station', verbose_name=_(u'Откуда'), related_name="promosearch_scheduletable_from")
    station_to = models.ForeignKey('www.Station', verbose_name=_(u'Куда'), related_name="promosearch_scheduletable_to")
    value = models.CharField(verbose_name=_(u'Текст'), max_length=255)

    def __unicode__(self):
        return "%s: %s-%s" % (self.company.L_title(), self.station_from.L_title(), self.station_to.L_title())

    class Meta:
        verbose_name = _(u'Таблица соответствия "поставщик+направление" -> "расписание"')
        ordering = ('promo_page', 'company', 'station_from')
        app_label = 'promosearch'


class TariffTable(models.Model):
    """
    RASP-13600 Таблица соответствия для цен на ЖД. "Поставщик+Направление" -> "Цена". Идея в том, что мы в динамике
    без опроса УФС, не сможем показывать адекватные цены по всем направлениям, а контент-менеджеры смогут указывать
    правильную минимальную цену.
    """

    promo_page = models.ForeignKey('promosearch.PromoPage', verbose_name=_(u'Промо-страница'))
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), default=1)
    company = models.ForeignKey('www.Company', verbose_name=_(u'Поставщик'))
    station_from = models.ForeignKey('www.Station', verbose_name=_(u'Откуда'), related_name="promosearch_tarifftable_from")
    station_to = models.ForeignKey('www.Station', verbose_name=_(u'Куда'), related_name="promosearch_tarifftable_to")
    value = models.CharField(verbose_name=_(u'Цена'), max_length=255)

    def __unicode__(self):
        return "%s: %s-%s" % (self.company.L_title(), self.station_from.L_title(), self.station_to.L_title())

    class Meta:
        verbose_name = _(u'Таблица соответствия "поставщик+направление" -> "цена"')
        ordering = ('promo_page', 'company', 'station_from')
        app_label = 'promosearch'


class TariffCache(models.Model):
    """
    Хранит цены для рейсов самолетов. Заполняется по cron-у
    """
    route_number = models.CharField(verbose_name=_(u"Номер рейс"), max_length=100, null=False, default='')
    station_from = models.ForeignKey('www.Station', verbose_name=_(u'Откуда'), related_name="promosearch_tariffcache_from")
    station_to = models.ForeignKey('www.Station', verbose_name=_(u'Куда'), related_name="promosearch_tariffcache_to")
    value_min = models.FloatField(verbose_name=_(u'Цена min'))
    value_max = models.FloatField(verbose_name=_(u'Цена max'))
