#! -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
import argparse
import time
from itertools import product

from django.db import transaction
from django.db.models import Q
from django.db import connection

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.schedule import Supplier, RThread
from travel.avia.library.python.common.models.tariffs import UFSDepthOfSales, UFSDirectionDaysAhead
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils.unicode_csv import UnicodeWriter
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
from travel.avia.admin.lib.tmpfiles import clean_temp, get_tmp_filepath
from travel.avia.admin.www.utils.common import RegExpDict


log = logging.getLogger(__name__)


class UFSDirectionDaysAheadBuilder(object):
    @transaction.atomic
    @clean_temp
    def run(self):
        t_start = time.time()

        log.info(u'Для направлений заполняем количество дней, на которые можно спрашивать цены от УФС.')

        self.delete_old_data()

        ufs_directions_days_ahead = self.get_ufs_directions_days_ahead()

        self.save_directions(ufs_directions_days_ahead)

        log.info(u'Заполнили количество дней для запроса цен от УФС по направлениям за %s сек.',
                 str(int(time.time() - t_start)))

    def get_ufs_directions_days_ahead(self):
        log.info(u'Собираем данные точка-точка-дни...')

        ufs_directions_days_ahead = {}
        ufs_depth_of_sales = RegExpDict({d.number: d.days_ahead for d in UFSDepthOfSales.objects.all()})

        tis_id = Supplier.objects.get(code='tis').id
        for thread in RThread.objects.filter(supplier_id=tis_id).exclude(number=u''):
            days_ahead = ufs_depth_of_sales[thread.number]
            if days_ahead is None:
                continue

            thread_stations = thread.get_stations()
            if len(thread_stations) < 2:
                continue

            express_codes = [self.get_express_codes(station) for station in thread_stations]

            for i, from_codes in enumerate(express_codes[:-1]):
                for to_codes in express_codes[i + 1:]:

                    for key in product(from_codes, to_codes):
                        old_days_ahead = ufs_directions_days_ahead.setdefault(key, days_ahead)

                        if days_ahead > old_days_ahead:
                            ufs_directions_days_ahead[key] = days_ahead

        log.info(u'Собрали данные точка-точка-дни. Всего %s.', len(ufs_directions_days_ahead))

        return ufs_directions_days_ahead

    @cache_method_result
    def get_express_codes(self, station):
        express_codes = set()

        if station.express_id:
            express_codes.add(station.express_id)

        # еще нужно взять 'спецстанцию для экспресс' для данного населенного пункта
        if station.settlement_id:
            codes = self.get_main_express_codes_in_settelment(station.settlement)

            express_codes.update(codes)

        return express_codes

    @cache_method_result
    def get_main_express_codes_in_settelment(self, settelment):
        stations = Station.objects.filter(Q(settlement=settelment) | Q(station2settlement__settlement=settelment))\
            .filter(majority__code='express_fake').distinct().only('express_id')

        express_codes = [s.express_id for s in stations if s.express_id]

        return express_codes

    def delete_old_data(self):
        log.info(u'Удаляем старые данные о количестве дней, на которые можно спрашивать цены от УФС...')

        sql = ur'TRUNCATE tariffs_ufsdirectiondaysahead'

        cursor = connection.cursor()
        cursor.execute(sql)

        log.info(u'Удалили старые данные.')

    def save_directions(self, ufs_directions_days_ahead):
        log.info(u'Записываем собранные данные...')

        tmp_file_name = get_tmp_filepath('ufs_directions_days_ahead.tsv')

        self.write_directions_to_file(ufs_directions_days_ahead, tmp_file_name)

        self.save_directions_to_db(tmp_file_name)

        log.info(u'Записали собранные данные. Всего %s.', UFSDirectionDaysAhead.objects.count())

    def save_directions_to_db(self, filename):
        columns = ['from_express_code', 'to_express_code', 'days_ahead']
        columns = u','.join(columns)

        sql = ur"""LOAD DATA LOCAL INFILE %s
INTO TABLE tariffs_ufsdirectiondaysahead
CHARACTER SET utf8
FIELDS ENCLOSED BY '"' ({columns})""".format(columns=columns)

        cursor = connection.cursor()
        cursor.execute(sql, [filename])

    def write_directions_to_file(self, ufs_directions_days_ahead, tmp_file_name):
        with open(tmp_file_name, 'w') as tmp_file:
            csv_writer = UnicodeWriter(tmp_file, delimiter='\t', lineterminator='\n')

            for (from_code, to_code), days_ahead in ufs_directions_days_ahead.iteritems():
                csv_writer.writerow([from_code, to_code, str(days_ahead)])


def main():
    parser = argparse.ArgumentParser()

    parser.add_argument('-v', '--verbose', action='store_true', help='increase output verbosity')

    args = parser.parse_args()

    if args.verbose:
        print_log_to_stdout(log)

    create_current_file_run_log()

    UFSDirectionDaysAheadBuilder().run()
