# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import sys
import unittest
from datetime import date, time, datetime

from travel.avia.library.python.common.models.geo import Settlement, Station, Region, Country, CityMajority
from travel.avia.library.python.common.models.schedule import RThread, Route
from travel.avia.library.python.common.utils.date import get_local_time, timedelta2hours

_encoding = 'utf8'


def enc(data):
    return data.encode(_encoding, 'ignore')


class RaspTestCase(unittest.TestCase):
    def shortDescription(self):
        doc = super(RaspTestCase, self).shortDescription()
        return doc and enc(doc) or None


class TestBaseIntegrity(RaspTestCase):
    nose_skip_class = True
    _object_set = {}
    _region_set = {}
    _country_set = {}

    @classmethod
    def _get_object_set(cls, model, **kwargs):
        key = model.__name__ + ':'.join([key + '=' + repr(value) for key, value in kwargs.iteritems()])
        try:
            return cls._object_set[key]
        except KeyError:
            try:
                objs = model.objects.filter(**kwargs)
            except AttributeError:
                objs = model.objects.filter(**kwargs)
            ids = set([obj.id for obj in objs])
            cls._object_set[key] = ids
            return ids

    @classmethod
    def get_settlement_set(cls, **kwargs):
        return cls._get_object_set(Settlement, **kwargs)

    @classmethod
    def get_region_set(cls, **kwargs):
        return cls._get_object_set(Region, **kwargs)

    @classmethod
    def get_country_set(cls, **kwargs):
        return cls._get_object_set(Country, **kwargs)

    @classmethod
    def get_rthread_set(cls, **kwargs):
        return cls._get_object_set(RThread, **kwargs)

    @classmethod
    def get_route_set(cls, **kwargs):
        return cls._get_object_set(Route, **kwargs)

    def testStationGeo(self):
        u"""Проверяем, что все станции привязанны к существующим городам, регионам и странам
        """
        settlement_set = TestBaseIntegrity.get_settlement_set()
        region_set = TestBaseIntegrity.get_region_set()
        country_set = TestBaseIntegrity.get_country_set()
        stations = Station.objects.all()
        bad_settlement_stations = []
        bad_region_stations = []
        bad_country_stations = []
        for station in stations:
            if (
                station.settlement_id
                and station.settlement_id not in settlement_set
            ):
                bad_settlement_stations.append(station)
            if (
                station.region_id
                and station.region_id not in region_set
            ):
                bad_region_stations.append(station)
            if (
                station.country_id
                and station.country_id not in country_set
            ):
                bad_country_stations.append(station)

        if (
            bad_settlement_stations
            or bad_region_stations
            or bad_country_stations
        ):
            msg = u"Станции привязанные к несуществующим городам:\n"
            if bad_settlement_stations:
                msg += u'\n'.join(
                [unicode(station.id) + u' ' + unicode(station.title)
                 for station in bad_settlement_stations])
            else:
                msg += u"Нет"
            msg += u"\n\n"
            msg += u"Станции привязанные к несуществующим регионам:\n"
            if bad_region_stations:
                msg += u'\n'.join(
                [unicode(station.id) + u' ' + unicode(station.title)
                 for station in bad_region_stations])
            else:
                msg += u"Нет"
            msg += u"\n\n"
            msg += u"Станции привязанные к несуществующим странам:\n"
            if bad_country_stations:
                msg += u'\n'.join(
                [unicode(station.id) + u' ' + unicode(station.title)
                 for station in bad_country_stations])
            else:
                msg += u"Нет"

            self.fail(enc(msg))

    def testSettlementGeo(self):
        u"""Проверяем, что все города привязаны к существующим регионам и странам
        """
        region_set = TestBaseIntegrity.get_region_set()
        country_set = TestBaseIntegrity.get_country_set()
        settlements = Settlement.objects.all()
        bad_region_settlements = []
        bad_country_settlements = []
        for settlement in settlements:
            if (
                settlement.region_id
                and settlement.region_id not in region_set
            ):
                bad_region_settlements.append(settlement)
            if (
                settlement.country_id
                and settlement.country_id not in country_set
            ):
                bad_country_settlements.append(settlement)

        if bad_region_settlements or bad_country_settlements:
            msg = u"Города привязанные к несуществующим регионам:\n"
            if bad_region_settlements:
                msg += u'\n'.join(
                [unicode(settlement.id) + u' ' + unicode(settlement.title)
                 for settlement in bad_region_settlements])
            else:
                msg += u"Нет"
            msg += u"\n\n"
            msg += u"Города привязанные к несуществующим странам:\n"
            if bad_country_settlements:
                msg += u'\n'.join(
                [unicode(settlement.id) + u' ' + unicode(settlement.title)
                 for settlement in bad_country_settlements])
            else:
                msg += u"Нет"

            self.fail(enc(msg))

    def testTimeZoneHierarchy(self):
        u"""Проверяем иерархию часовых поясов
        Что часовой пояс станции совпадает с часовым поясом региона или города,
        к которому привязана станция. И что часовой пояс города совпадает
        с часовым поясом региона, которому привязан город.
        """
        def compare_tz(tz1, tz2):
            """
            @param tz1:
            @param tz2:

            @return: tuple(bool has_diff, unicode diff_string)
            has_diff - Совпадает или нет;
            diff_string - строка сравнения временных зон.
            """
            self.assertTrue(tz1 and tz2,
                            u"В функцию сравнения переданы неправильные временные зоны %s %s" % (tz1, tz2))
            if tz1 == tz2:
                return False, None
            year = date.today().year
            summer_date = date(year, 7, 22)
            winter_date = date(year, 1, 1)
            msk_summer_time = datetime.combine(summer_date, time(0, 0, 0))
            summer1 = get_local_time(tz1, msk_summer_time).replace(tzinfo=None)
            summer2 = get_local_time(tz2, msk_summer_time).replace(tzinfo=None)

            msk_winter_time = datetime.combine(winter_date, time(0, 0, 0))
            winter1 = get_local_time(tz1, msk_winter_time).replace(tzinfo=None)
            winter2 = get_local_time(tz2, msk_winter_time).replace(tzinfo=None)

            if summer1 != summer2 or winter1 != winter2:
                summer_diff = timedelta2hours(summer1 - summer2)
                winter_diff = timedelta2hours(winter1 - winter2)
                if summer_diff != winter_diff:
                    return True, u"зоны %s и %s отличаются зимой на %.2f часа, летом на %.2f часа" % \
                        (tz1, tz2, winter_diff, summer_diff)
                else:
                    return True, u"зоны %s и %s отличаются на %.2f часа" % \
                        (tz1, tz2, winter_diff)
            return False, None

        bad_stations = []
        bad_settlements = []
        errors = []
        # Для начала обработаем все станции, у которых должен быть регион.
        for station in Station.objects.filter(hidden=False,
                                              country__in=Country.mast_have_region_countries(),
                                              settlement__isnull=True).\
                                       select_related('region', 't_type'):
            if not station.region:
                bad_stations.append((station, u"У станции в стране %s отсутствует регион" % station.country.title))
                continue
            if not station.time_zone:
                bad_stations.append((station, u"У станции в стране %s регионе %s отсутствует временная зона" %
                                     (station.country.title, station.region.title)))
                continue
            if not station.region.time_zone:
                errors.append(u"У региона %s отсутствует временная зона" % station.region.title)
                continue
            has_diff, diff_str = compare_tz(station.time_zone, station.region.time_zone)
            if has_diff:
                bad_stations.append((station, u"Расхождение с регионом %s: %s" %
                                     (station.region.title, diff_str)))
        for station in Station.objects.filter(hidden=False, country__isnull=True):
            bad_stations.append((station, u"Отсутствует привязка к стране"))
        # Сверяем зону остальных станций, с зоной столицы
        for station in Station.objects.exclude(country__in=Country.mast_have_region_countries()).\
                       filter(country__isnull=False, settlement__isnull=True,
                              hidden=False).\
                       select_related('country', 't_type'):
            capital = station.country.get_capital()
            if capital is None:
                errors.append(u"В стране %s нет столицы" % station.country.title)
                continue
            if not capital.time_zone:
                errors.append(u"В столице %s %s страны %s не указана временная зона" % (capital.id, capital.title, station.country.title))
                continue
            if not station.time_zone:
                bad_stations.append((station, u"У станции в стране %s отсутствует временная зона" %
                                     station.country.title))
                continue
            has_diff, diff_str = compare_tz(station.time_zone, capital.time_zone)
            if has_diff:
                bad_stations.append((station, u"Расхождение состолицей %s: %s" %
                                     (capital.title, diff_str)))

        # Для начала обработаем все города, у которых должен быть регион.
        for settlement in Settlement.objects.filter(
            hidden=False,
            country__in=Country.mast_have_region_countries()
        ).select_related('region'):
            if not settlement.region:
                bad_settlements.append((settlement, u"У города в стране %s отсутствует регион" % station.country.title))
                continue
            if not settlement.time_zone:
                bad_settlements.append((settlement, u"У города в стране %s регионе %s отсутствует временная зона" %
                                        (settlement.country.title, settlement.region.title)))
                continue
            if not settlement.region.time_zone:
                errors.append(u"У региона %s отсутствует временная зона" % settlement.region.title)
                continue
            has_diff, diff_str = compare_tz(settlement.time_zone, settlement.region.time_zone)
            if has_diff:
                bad_settlements.append((settlement, u"Расхождение с регионом %s: %s" % (settlement.region.title, diff_str)))
        for settlement in Settlement.objects.filter(hidden=False, country__isnull=True):
            bad_settlements.append((settlement, u"Отсутствует привязка к стране"))
        # Сверяем зону остальных городов, с зоной столицы
        for settlement in Settlement.objects.exclude(country__in=Country.mast_have_region_countries()).\
                       filter(country__isnull=False, hidden=False).\
                       select_related('country'):
            capital = settlement.country.get_capital()
            if capital is None:
                errors.append(u"В стране %s нет столицы" % settlement.country.title)
                continue
            if not capital.time_zone:
                errors.append(u"В столице %s %s страны %s не указана временная зона" % (capital.id, capital.title, settlement.country.title))
                continue
            if not settlement.time_zone:
                bad_settlements.append((settlement, u"У города в стране %s отсутствует временная зона" %
                                        settlement.country.title))
                continue
            has_diff, diff_str = compare_tz(settlement.time_zone, capital.time_zone)
            if has_diff:
                bad_settlements.append((settlement, u"Расхождение состолицей %s: %s" %
                                        (capital.title, diff_str)))

        bad_stations = sorted(list(set(bad_stations)), key=lambda x: (x[1], x[0].title))
        bad_settlements = sorted(list(set(bad_settlements)), key=lambda x: (x[1], x[0].title))
        errors = sorted(list(set(errors)))
        msg = u"\n"
        if errors:
            msg += u"\n".join(errors)
            msg += u"\n\n"
        if bad_stations:
            bad_stations = [u"Станция %s %s %s: %s" %
                            (station.id, station.title,
                             station.t_type.title_ru, message)
                            for station, message in bad_stations]
            msg += u"\n".join(bad_stations)
            msg += u"\n\n"
        if bad_settlements:
            bad_settlements = [u"Город %s %s: %s" % (settlement.id, settlement.title, message)
                               for settlement, message in bad_settlements]
            msg += u"\n".join(bad_settlements)
            msg += u"\n\n"

        msg = msg.rstrip()
        if msg.strip():
            self.fail(enc(msg))


class TestCapitols(RaspTestCase):
    nose_skip_class = True

    def testRegionCapitol(self):
        u"""Проверка столиц регионов"""
        regions = Region.objects.all()
        majority = CityMajority.objects.get(title=u"Столица области")
        msgs = []
        for region in regions:
            capitols = Settlement.objects.filter(majority__id__lte=majority.id, region=region).count()
            if capitols == 0:
                msgs.append(u"У области %d %s нет столицы" % (region.id, region.title))
            elif capitols > 1:
                msgs.append(u"У области %d %s несколько столиц" % (region.id, region.title))

        if msgs:
            self.fail(enc(u"\n" + u"\n".join(msgs)))

    def testCountryCapitol(self):
        u"""Проверка столиц стран"""
        countries = Country.objects.all()
        majority = CityMajority.objects.get(title=u"Столица страны")
        msgs = []
        for country in countries:
            capitols = Settlement.objects.filter(majority__id__lte=majority.id, country=country).count()
            if capitols == 0:
                msgs.append(u"У страны %d %s нет столицы" % (country.id, country.title))
            elif capitols > 1:
                msgs.append(u"У страны %d %s несколько столиц" % (country.id, country.title))

        if msgs:
            self.fail(enc(u"\n" + u"\n".join(msgs)))

# debug = unittest.TestSuite([TestBaseIntegrity('testRegionCapitol')])


def main():
    if len(sys.argv) >= 2:
        unittest.main(defaultTest=sys.argv[1])
    unittest.main()
