# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import codecs
import json
import logging
import os.path
import time as os_time
from datetime import timedelta, datetime, time
from functools import wraps
from optparse import OptionParser

from django.conf import settings

from travel.avia.library.python.common.models.geo import ExternalDirection, Station
from travel.avia.library.python.common.models_utils.i18n import RouteLTitle
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import timedelta_to_str_hours_and_minutes
from travel.avia.library.python.common.utils.unicode_csv import UnicodeDictWriter
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
from travel.avia.library.python.route_search.models import ZNodeRoute2
from travel.avia.library.python.route_search.shortcuts import find


log = logging.getLogger(__name__)


FROM_CENTER_TSV_FILEPATH = os.path.join(settings.EXPORT_PATH, 'nearest_suburban_from_center_2.tsv')
TO_CENTER_TSV_FILEPATH = os.path.join(settings.EXPORT_PATH, 'nearest_suburban_to_center_2.tsv')
TO_FROM_CENTER_JSON_FILEPATH = os.path.join(settings.EXPORT_PATH, 'nearest_suburban_to_from_center_2.json')

STATIONS_TSV_FILEPATH = os.path.join(settings.EXPORT_PATH, 'export_nearest_suburban_stations.tsv')
STATIONS_JSON_FILEPATH = os.path.join(settings.EXPORT_PATH, 'export_nearest_suburban_stations.json')


def log_work_time(log):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            func_name = func.__name__
            module_name = func.__module__
            if module_name != '__main__':
                func_name = module_name + '.' + func_name

            start_time = os_time.time()
            result = func(*args, **kwargs)
            log.info(u'%s отработала за %.2f', func_name,
                     os_time.time() - start_time)

            return result

        return wrapper

    return decorator


@log_work_time(log)
def generate_files(now_aware):
    generate_export_stations_from_db()

    stations_data = json.loads(open(STATIONS_JSON_FILEPATH).read())

    with ToFromCenterTsvWriter(FROM_CENTER_TSV_FILEPATH) as tsv_from_center, \
            ToFromCenterTsvWriter(TO_CENTER_TSV_FILEPATH) as tsv_to_center, \
            ToFromCenterJsonWriter(TO_FROM_CENTER_JSON_FILEPATH) as json_to_from_center:
        for data in stations_data.itervalues():
            start = os_time.time()

            center = Station.objects.get(id=data['cst']['id'])
            station = Station.objects.get(id=data['st']['id'])

            log.info(u'Обрабатываем %s - %s', station.title, center.title)

            nearest_from_center = generate_nearest_fromto(center, station, now_aware)
            nearest_to_center = generate_nearest_fromto(station, center, now_aware)

            tsv_from_center.write_segments(station, nearest_from_center)
            tsv_to_center.write_segments(station, nearest_to_center)

            json_to_from_center.write_segments(station, nearest_to_center, nearest_from_center)

            log.info(u'per record time %f', os_time.time() - start)


def generate_export_stations_from_db():
    added_cities = set()

    def can_use_station(station):
        if not station.settlement:
            log.info(u'Станция без города пропускаем %s %s', station.id, station.title)
            return False

        if not station.settlement.suburban_zone:
            log.info(u'Не нашли зоны города станции пропускаем %s %s', station.id, station.title)
            return False

        if not station.settlement._geo_id:
            log.info(u'У города не указан geo_id пропускаем %s %s', station.id, station.title)
            return False

        # Совпадает с центром одной из зон
        if ExternalDirection.objects.filter(base_station=station):
            log.info(u'Cтанция совпадает с центром одной из зон пропускаем %s %s', station.id, station.title)
            return False

        # Совпадает с центром одной из зон
        if station.settlement.majority_id in (1, 2):
            log.info(u'Cтанция находится в столице или областном центре пропускаем %s %s',
                     station.id, station.title)
            return False

        return True

    with StationJsonWriter(STATIONS_JSON_FILEPATH) as stations_json_writer, \
            StationTsvWriter(STATIONS_TSV_FILEPATH, encoding=None) as stations_tsv_writer:
        all_stations = Station.objects.filter(t_type__code='train').order_by('majority__id')

        stations = [s for s in all_stations if can_use_station(s)]

        stations.sort(key=lambda s: 0 if s.country_id == 225 else 10)

        for station in stations:
            zone = station.settlement.suburban_zone

            try:
                base_station = ExternalDirection.objects.filter(
                    externaldirectionmarker__station=station,
                    suburban_zone=zone
                )[0].base_station

            except IndexError:
                log.info(u'Не нашли базовой станции пропускаем %s %s', station.id, station.title)

                continue

            if station.settlement in added_cities:
                log.info(u'Город уже обрабатывали %s %s пропускаем %s %s', station.settlement.id, station.settlement.title,
                         station.id, station.title)
                continue

            if not has_suburban_service(station, base_station):
                log.info(u'Нет сообщения от до центра %s %s пропускаем %s %s', base_station.id,
                         base_station.title, station.id, station.title)
                continue

            added_cities.add(station.settlement)

            log.info(u'Добавили станцию %s %s %s с центром %s %s', station.title, station.id,
                     station.settlement._geo_id, base_station.title, base_station.id)

            stations_tsv_writer.write_station(station, base_station)
            stations_json_writer.write_station(station, base_station)

        log.info(u'Сгенерированы файлы станций из всех станций в базе: %s, %s',
                 STATIONS_TSV_FILEPATH, STATIONS_JSON_FILEPATH)


def has_suburban_service(station1, station2):
    qs1 = ZNodeRoute2.objects.filter(station_from=station1, station_to=station2, thread__t_type__code='suburban')
    qs2 = ZNodeRoute2.objects.filter(station_from=station2, station_to=station1, thread__t_type__code='suburban')

    return qs1.exists() or qs2.exists()


def generate_nearest_fromto(station_from, station_to, now_aware):
    local_today = now_aware.astimezone(station_from.pytz).date()

    # Ищем на 2 дня вперед
    end_limit = datetime.combine(local_today, time(0, 0)) + timedelta(2)
    end_limit = station_from.pytz.localize(end_limit)

    half_hour = timedelta(minutes=30)
    search = find(station_from, station_to, local_today, 'suburban')

    reach_time_barrier = False
    obtain_extra_records = False
    extra_count = 0
    nearest_segments = []
    first_start = None

    for segment in search:
        if segment.departure >= end_limit:
            break

        segment_to_append = None

        if reach_time_barrier and obtain_extra_records:
            break

        if segment.departure <= now_aware:
            continue

        if first_start is not None and \
           segment.departure > first_start + timedelta(hours=23, minutes=59):
            continue

        if not reach_time_barrier and segment.departure < (now_aware + half_hour):
            segment_to_append = segment
        else:
            reach_time_barrier = True

        if reach_time_barrier and extra_count < 10:
            segment_to_append = segment
            extra_count += 1
        elif reach_time_barrier:
            obtain_extra_records = True

        if segment_to_append:
            if first_start is None:
                first_start = segment_to_append.departure

            nearest_segments.append(segment_to_append)

    log.info(u'Попало в нужный диапазон %s электричек', len(nearest_segments))
    return nearest_segments


class TransactionFileHandler(object):
    SUFFIX = '.inprocess'

    def __init__(self, filename, mode='w', encoding='utf-8'):
        self.filename = filename
        self.tmp_filename = filename + self.SUFFIX
        self.mode = mode
        self.encoding = encoding

        self.handler = None

    def __enter__(self):
        self.handler = codecs.open(self.tmp_filename, self.mode, self.encoding)

        return self

    def __exit__(self, exception_type, exception_value, traceback):
        if self.handler is not None:
            self.handler.close()

        if exception_type is not None:
            self.on_error()

        else:
            # Только если не было exception заменяяем старый файл на новый
            try:
                os.rename(self.tmp_filename, self.filename)

            except OSError:
                self.on_error()

    def on_error(self):
        log.error(u'В процессе генерации файла %s произошла ошибка. '
                  u'Оставляем старую версию файла', self.filename)

        try:
            os.remove(self.tmp_filename)

        except OSError:
            pass


class ToFromCenterTsvWriter(TransactionFileHandler):
    def write_segments(self, station, segments):
        start_record = unicode(station.id) + u'\t'
        other_records = self.export_tsv_segments(segments, self.export_tsv_with_uid)

        self.handler.write(start_record + other_records + u'\n')

    @staticmethod
    def export_tsv_segments(segments, export_function):
        results = [export_function(segment) for segment in segments]

        return u'\t'.join(map(lambda x: u'\t'.join(x), results))

    @staticmethod
    def export_tsv_with_uid(segment):
        return (
            segment.departure.strftime('%Y%m%d%H%M%S'),
            segment.thread.title_short or segment.thread.title or u'',
            unicode(int(segment.thread.express_type == 'express')),
            segment.thread.uid or u''
        )


class ToFromCenterJsonWriter(TransactionFileHandler):
    def __init__(self, filename, mode='w', encoding='utf-8'):
        super(ToFromCenterJsonWriter, self).__init__(filename, mode, encoding)

        self.json_results = dict()

    def write_segments(self, station, nearest_to_center, nearest_from_center):
        self.json_results[station.id] = {
            'tc': map(self.export_json_with_uid, nearest_to_center),
            'fc': map(self.export_json_with_uid, nearest_from_center),
        }

    def __exit__(self, exception_type, exception_value, traceback):
        if self.handler is not None:
            self.handler.write(json.dumps(self.json_results, ensure_ascii=False, indent=4, encoding='utf-8'))

        super(ToFromCenterJsonWriter, self).__exit__(exception_type, exception_value, traceback)

    @staticmethod
    def export_json_with_uid(segment):
        RouteLTitle.fetch([segment.thread.L_title])

        # Электрички не могут идти больше суток, поэтому укладываемся в формат HH:MM
        travel_time = timedelta_to_str_hours_and_minutes(segment.duration)

        data = {
            'time': segment.departure.strftime('%Y%m%d%H%M%S'),
            'name': {
                'ru': segment.thread.L_title(lang='ru', short=True),
                'uk': segment.thread.L_title(lang='uk', short=True),
                'tr': segment.thread.L_title(lang='tr', short=True),
            },
            'uid': segment.thread.uid or u'',
            'travel_time': travel_time,
        }
        if segment.thread.express_type == 'express':
            data['express'] = 1

        return data


class StationTsvWriter(TransactionFileHandler):
    EXPORT_STATION_TSV_FIELDS = (
        'geo_id', 'station_id', 'station_title',
        'transport_center_station_id', 'transport_center_station_title',
        'station_title_uk', 'transport_center_station_title_uk'
    )

    def __enter__(self):
        super(StationTsvWriter, self).__enter__()

        self.station_writer = UnicodeDictWriter(
            self.handler,
            self.EXPORT_STATION_TSV_FIELDS,
            encoding='utf-8',
            delimiter='\t'
        )

        return self

    def write_station(self, station, base_station):
        self.station_writer.writerow({
            'geo_id': station.settlement._geo_id,
            'station_id': station.id,
            'station_title': station.L_title(lang='ru'),
            'transport_center_station_id': base_station.id,
            'transport_center_station_title': base_station.L_title(lang='ru'),
            'station_title_uk': station.L_title(lang='uk'),
            'transport_center_station_title_uk': base_station.L_title(lang='uk'),
        })


class StationJsonWriter(TransactionFileHandler):
    def __init__(self, filename, mode='w', encoding='utf-8'):
        super(StationJsonWriter, self).__init__(filename, mode, encoding)

        self.json_results = dict()

    def write_station(self, station, base_station):
        self.json_results[station.settlement._geo_id] = {
            'st': {
                'id': station.id,
                'name': {
                    'ru': station.L_title(lang='ru'),
                    'uk': station.L_title(lang='uk'),
                    'tr': station.L_title(lang='tr'),
                }
            },
            'cst': {
                'id': base_station.id,
                'name': {
                    'ru': base_station.L_title(lang='ru'),
                    'uk': base_station.L_title(lang='uk'),
                    'tr': base_station.L_title(lang='tr'),
                }
            },
        }

    def __exit__(self, exception_type, exception_value, traceback):
        if self.handler is not None:
            self.handler.write(json.dumps(self.json_results, ensure_ascii=False, indent=4, encoding='utf-8'))

        super(StationJsonWriter, self).__exit__(exception_type, exception_value, traceback)


usage = u'Usage: python %prog [options]'


def main():
    optparser = OptionParser(usage=usage, description=__doc__)
    optparser.add_option('-v', '--verbose', action='store_true',
                         help=u'выводить лог на экран')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    create_current_file_run_log()

    action = 'generate-files'
    if args:
        action = args[0]

    if action == 'generate-files':
        generate_files(environment.now_aware())

    elif action == 'generate-station-file-from-db':
        generate_export_stations_from_db()
