# -*- coding: utf-8 -*-

"""
RASPADMIN-793 Генерация файла для мобильной морды
В файле должны быть geoid и актуальные виды транспорта
"""

import travel.avia.admin.init_project  # noqa

import json
import logging
import os.path
from collections import defaultdict

from django.conf import settings

from travel.avia.library.python.common.models.geo import Settlement, Station, Station2Settlement, StationMajority
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log


log = logging.getLogger(__name__)


OUTPUT_FILEPATH = os.path.join(settings.EXPORT_PATH, 't_types_by_geoid.json')
TRAIN_AND_SUBURBAN_IDS = {TransportType.TRAIN_ID, TransportType.SUBURBAN_ID}


def generate_t_types_by_geoid():
    cities_query = Settlement.hidden_manager.filter(_geo_id__isnull=False)
    geoid_by_city_ids = {id_: geoid for id_, geoid in cities_query.values_list('id', '_geo_id')}
    log.info(u'Всего нужно обработать %s населенных пунктов', len(geoid_by_city_ids))

    t_type_ids_by_city_ids = {}
    for chunk_city_ids in chunker(geoid_by_city_ids.keys(), chunk_size=3000):
        t_type_ids_by_city_ids.update(get_t_type_ids_by_city_ids(chunk_city_ids))
    log.info(u'Населенных пунктов с известным расписанием %s', len(t_type_ids_by_city_ids))

    add_extra_t_type_ids(t_type_ids_by_city_ids)
    add_suburban(t_type_ids_by_city_ids)

    result = {
        geoid_by_city_ids[city_id]: {
            'train': has_train(t_type_ids),
            'suburban': has_suburban(t_type_ids),
            'plane': has_plane(t_type_ids),
            'bus': has_bus(t_type_ids),
            'water': has_water(t_type_ids),
        }
        for city_id, t_type_ids in t_type_ids_by_city_ids.iteritems() if t_type_ids
    }

    return result


def get_extra_t_type_ids_by_city_ids():
    station_ids = set(Station2Settlement.objects.all().values_list('station_id', flat=True))
    visible_t_type_ids_by_station_ids = {
        id_: t_type_id
        for id_, t_type_id in Station.hidden_manager
                                     .filter(id__in=station_ids, majority__id__lte=StationMajority.IN_TABLO_ID)
                                     .values_list('id', 't_type_id')
    }

    extra_t_type_ids_by_city_ids = defaultdict(set)
    city_and_station_query = Station2Settlement.objects.filter(station_id__in=visible_t_type_ids_by_station_ids.keys())\
                                                       .values_list('settlement_id', 'station_id')
    for city_id, station_id in city_and_station_query:
        extra_t_type_ids_by_city_ids[city_id].add(visible_t_type_ids_by_station_ids[station_id])

    return extra_t_type_ids_by_city_ids


def get_t_type_ids_by_city_ids(city_ids):
    t_type_ids_by_city_ids = defaultdict(set)
    city_and_station_query = Station.hidden_manager.filter(settlement_id__in=city_ids) \
                                                   .filter(majority__id__lte=StationMajority.IN_TABLO_ID) \
                                                   .values_list('settlement_id', 't_type_id')
    for city_id, t_type_id in city_and_station_query:
        t_type_ids_by_city_ids[city_id].add(t_type_id)

    return t_type_ids_by_city_ids


def add_extra_t_type_ids(t_type_ids_by_city_ids):
    extra_t_type_ids_by_city_ids = get_extra_t_type_ids_by_city_ids()

    for city_id in t_type_ids_by_city_ids:
        t_type_ids_by_city_ids[city_id] |= extra_t_type_ids_by_city_ids[city_id]


def add_suburban(t_type_ids_by_city_ids):
    city_ids_with_train = [city_id for city_id, t_type_ids in t_type_ids_by_city_ids.iteritems()
                           if TransportType.TRAIN_ID in t_type_ids]
    log.info(u'Нужно отделить электрички от поездов для %s населенных пунктов', len(city_ids_with_train))

    city_ids_with_suburban_threads = set(Settlement.objects.filter(
        station__rtstation__thread__t_type=TransportType.SUBURBAN_ID
    ).values_list('id', flat=True).distinct())

    city_ids_with_train_threads = set(Settlement.objects.filter(
        station__rtstation__thread__t_type=TransportType.TRAIN_ID
    ).values_list('id', flat=True).distinct())

    for city_id in city_ids_with_train:
        if city_id not in city_ids_with_train_threads:
            t_type_ids_by_city_ids[city_id] -= {TransportType.TRAIN_ID}

        if city_id in city_ids_with_suburban_threads:
            t_type_ids_by_city_ids[city_id] |= {TransportType.SUBURBAN_ID}


def has_train(t_type_ids):
    return TransportType.TRAIN_ID in t_type_ids


def has_suburban(t_type_ids):
    return TransportType.SUBURBAN_ID in t_type_ids


def has_plane(t_type_ids):
    return TransportType.PLANE_ID in t_type_ids


def has_bus(t_type_ids):
    return TransportType.BUS_ID in t_type_ids


def has_water(t_type_ids):
    return TransportType.WATER_ID in t_type_ids


def _main():
    log.info(u'Начинаем генерацию t_types_by_geoid.json')
    result = generate_t_types_by_geoid()
    with open(OUTPUT_FILEPATH, 'w') as fp:
        json.dump(result, fp)

    log.info(u'Успешно завершили генерацию t_types_by_geoid.json')


def main():
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument("-v", "--verbose", dest="verbose", action="store_true")
    args = parser.parse_args()

    create_current_file_run_log()
    if args.verbose:
        print_log_to_stdout()

    _main()
