# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
import os.path
from os import makedirs
from optparse import OptionParser
from datetime import date, datetime, time
from shutil import rmtree

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.utils.date import get_local_time, timedelta2hours
from travel.avia.library.python.common.utils.unicode_csv import UnicodeDictWriter
from travel.avia.library.python.common.lib.mail import mail_process_with_attaches
from travel.avia.admin.lib.tmpfiles import temp_manager
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log


script_name = os.path.basename(__file__)

log = logging.getLogger(__name__)

mimetype = 'text/csv; charset=utf-8'

tests = []

TMP_DIR = temp_manager.get_tmp_dir('geobase_consistency_tests')


def _main():
    try:
        if not os.path.exists(TMP_DIR):
            makedirs(TMP_DIR)

        files = []
        messages = []

        for test_func in tests:
            message, filepath = test_func()
            if message:
                messages.append(message)
                files.append((filepath, mimetype))

        subject = u"Правильность географической информации в базе"
        mail_message = u"\n".join(messages)
        mail_process_with_attaches(subject, mail_message, files=files)

    finally:
        rmtree(TMP_DIR, ignore_errors=True)


def test_station_and_regions_timezones():
    filename = 'station_regions_timezones.csv'
    message = u"Ошибки в сравнении временных зон станций и их регионов, данные в файле %s" % filename
    fieldnames = ('station_id', 'station_title', 'region_title', 'station_timezone', 'region_timezone', 'diff')

    filepath = os.path.join(TMP_DIR, filename)
    f = open(filepath, 'w')
    csv_writer = UnicodeDictWriter(f, fieldnames, delimiter=';', encoding='utf8')
    csv_writer.writehead()

    has_rows = False

    for station in Station.objects.all().select_related('region'):
        if station.region_id and not station.time_zone_not_check:
            if station.time_zone and station.region.time_zone:
                has_diff, diff = compare_tz(station.time_zone, station.region.time_zone)
            else:
                has_diff, diff = True, u"Зоны не заполнены у одного из объектов"

            if has_diff:
                rowdict = {
                    'station_id': station.id,
                    'station_title': station.title,
                    'region_title': station.region.title,
                    'station_timezone': station.time_zone,
                    'region_timezone': station.region.time_zone,
                    'diff': diff
                }

                csv_writer.writerow(rowdict)
                has_rows = True

    f.close()

    if has_rows:
        return message, filepath
    else:
        return None, None


def test_settlement_and_station_regions():
    filename = 'settlement_station_regions.csv'
    message = u"Ошибки в сравнении регионов городов станций и регионов станций, данные в файле %s" % filename
    fieldnames = ('station_id', 'station_title', 'station_region_title', 'settlement_id',
                  'settlement_title', 'settlement_region_title')

    filepath = os.path.join(TMP_DIR, filename)
    f = open(filepath, 'w')
    csv_writer = UnicodeDictWriter(f, fieldnames, delimiter=';', encoding='utf8')
    csv_writer.writehead()

    has_rows = False

    for station in Station.objects.all().select_related('settlement__region', 'region'):
        if station.settlement_id:
            if station.settlement.region_id != station.region_id:
                rowdict = {
                    'station_id': station.id,
                    'station_title': station.title,
                    'station_region_title': station.region_id and station.region.title,
                    'settlement_id': station.settlement_id,
                    'settlement_title': station.settlement.title,
                    'settlement_region_title': station.settlement.region_id and station.settlement.region.title,
                }

                csv_writer.writerow(rowdict)
                has_rows = True

    f.close()

    if has_rows:
        return message, filepath
    else:
        return None, None


def test_settlement_and_station_countries():
    filename = 'settlement_station_countries.csv'
    message = u"Ошибки в сравнении стран городов станций и стран станций, данные в файле %s" % filename
    fieldnames = ('station_id', 'station_title', 'station_country_title', 'settlement_id',
                  'settlement_title', 'settlement_country_title')

    filepath = os.path.join(TMP_DIR, filename)
    f = open(filepath, 'w')
    csv_writer = UnicodeDictWriter(f, fieldnames, delimiter=';', encoding='utf8')
    csv_writer.writehead()

    has_rows = False

    for station in Station.objects.all().select_related('settlement__country', 'country'):
        if station.settlement_id:
            if station.settlement.country_id != station.country_id:
                rowdict = {
                    'station_id': station.id,
                    'station_title': station.title,
                    'station_country_title': station.country_id and station.country.title,
                    'settlement_id': station.settlement_id,
                    'settlement_title': station.settlement.title,
                    'settlement_country_title': station.settlement.country_id and station.settlement.country.title,
                }

                csv_writer.writerow(rowdict)
                has_rows = True

    f.close()

    if has_rows:
        return message, filepath
    else:
        return None, None


def compare_tz(tz1, tz2):
    """
    @param tz1:
    @param tz2:

    @return: tuple(bool has_diff, unicode diff_string)
    has_diff - есть ли различие;
    diff_string - строка сравнения временных зон.
    """
    assert(tz1 and tz2)
    if tz1 == tz2:
        return False, None
    year = date.today().year
    summer_date = date(year, 07, 22)
    winter_date = date(year, 01, 01)
    msk_summer_time = datetime.combine(summer_date, time(0, 0, 0))
    summer1 = get_local_time(tz1, msk_summer_time).replace(tzinfo=None)
    summer2 = get_local_time(tz2, msk_summer_time).replace(tzinfo=None)

    msk_winter_time = datetime.combine(winter_date, time(0, 0, 0))
    winter1 = get_local_time(tz1, msk_winter_time).replace(tzinfo=None)
    winter2 = get_local_time(tz2, msk_winter_time).replace(tzinfo=None)

    if summer1 != summer2 or winter1 != winter2:
        summer_diff = timedelta2hours(summer1 - summer2)
        winter_diff = timedelta2hours(winter1 - winter2)
        if summer_diff != winter_diff:
            return True, u"зоны %s и %s отличаются зимой на %.2f часа, летом на %.2f часа" % \
                (tz1, tz2, winter_diff, summer_diff)
        else:
            return True, u"зоны %s и %s отличаются на %.2f часа" % \
                (tz1, tz2, winter_diff)
    return False, None


for name, value in locals().items():
    if name.startswith('test_') and callable(value):
        tests.append(value)


usage = u"Usage: python %prog [options]"


def main():
    optparser = OptionParser(usage=usage, description=__doc__)
    optparser.add_option('-v', '--verbose', action="store_true",
                         help=u"выводить лог на экран")
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    create_current_file_run_log()

    _main()
