# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
import os.path
from optparse import OptionParser
from datetime import datetime, date
from subprocess import check_call
from xml.etree import ElementTree
from xml.etree.ElementTree import iterparse

from django.conf import settings

from travel.avia.library.python.common.models.geo import Settlement, StationCode
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.special_tariffs import SpecialOffer
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.safe_xml_parser import get_xml_safe_parser
from travel.avia.admin.lib.fileutils import creating_dir_join
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_current_file_logger_name


script_name = os.path.basename(__file__)
log = logging.getLogger(__name__)

script_runs_log_name = 'special.script_runs.' + get_current_file_logger_name()
script_runs_log_dir = os.path.join(settings.LOG_PATH, *script_runs_log_name.split('.'))

wget_log_file = creating_dir_join(script_runs_log_dir, environment.now().strftime('wget_%Y-%m-%d_%H%M%S.log'))


def parse_date(text):
    try:
        return datetime.strptime(text, '%Y-%m-%d')

    except Exception:
        return None


def process_special_offer(offer_el):
    """ Return SpecialOffer list from offer_el prices """

    try:
        date_start = parse_date(offer_el.get('date_start')) or date.min
        date_end = parse_date(offer_el.get('date_end')) or date.max
        online_booking = bool(offer_el.get('online_booking'))
        text = offer_el.get('text')
        company_iata = offer_el.find('airline').get('code')

    except Exception:
        log.info(u'Skip special_offer tag %s', ElementTree.tostring(offer_el))

        return

    for price_el in offer_el.iter('price'):
        offer = SpecialOffer()

        offer.date_start = date_start
        offer.date_end = date_end

        offer.online_booking = online_booking
        offer.text = text

        offer.company_iata = company_iata

        offer.xml = ElementTree.tostring(price_el)

        try:
            offer.period_start = parse_date(price_el.get('date_start')) or date.min
            offer.period_end = parse_date(price_el.get('date_end')) or date.max

            offer.iata_from = price_el.get('depart')
            offer.iata_to = price_el.get('arrive')

            offer.tariff = float(price_el.get('price'))
            offer.service_class = price_el.get('service_class')
            offer.reverse = bool(int(price_el.get('roundtrip', 0)))
            offer.tariff_code = price_el.get('tariff_code')

        except Exception:
            log.info(u'Skip special_offer price tag %s', offer.xml)

            continue

        yield offer


def run():
    special_offers_file = os.path.join(
        settings.PROJECT_PATH,
        'www/db/scripts/data/special_offer_trevel.xml'
    )

    check_call([
        'wget', settings.SPECIAL_OFFERS_URL,
        '-O', special_offers_file,
        '-a', wget_log_file,
    ])

    offers = []

    for unused, elem in iterparse(special_offers_file, parser=get_xml_safe_parser()):
        if elem.tag == 'special_offer':
            offers.extend(process_special_offer(elem))

    log.info(u'Offers tags processed: %d', len(offers))

    iatas = [offer.iata_from for offer in offers] + [offer.iata_to for offer in offers]

    station_codes = StationCode.objects.filter(system__code='iata', code__in=iatas).\
        select_related('station', 'station__settlement')

    airports_by_iata = {
        c.code: c.station
        for c in station_codes
    }

    settlements_by_iata = {
        s.iata: s
        for s in Settlement.objects.filter(iata__in=iatas)
    }

    company_iatas = [offer.company_iata for offer in offers]

    # Из компаний с одинаковыми иата-кодами в словарь попадёт имеющая наибольший приоритет
    company_by_iata = {
        c.iata: c
        for c in Company.objects.filter(iata__in=company_iatas).order_by('-priority')
    }

    ready_offers = []

    for offer in offers:
        try:
            offer.airport_from = airports_by_iata.get(offer.iata_from)

            offer.settlement_from = offer.airport_from and \
                offer.airport_from.settlement or \
                settlements_by_iata.get(offer.iata_from)

        except ValueError:
            log.error('Not found iata [%s] xml: %s' % (offer.iata_from, offer.xml))

            continue

        try:
            offer.airport_to = airports_by_iata.get(offer.iata_to)

            offer.settlement_to = offer.airport_to and \
                offer.airport_to.settlement or \
                settlements_by_iata.get(offer.iata_to)

        except ValueError:
            log.error('Not found iata [%s] xml: %s' % (offer.iata_to, offer.xml.strip()))

            continue

        offer.company = company_by_iata[offer.company_iata]

        ready_offers.append(offer)

    if ready_offers:
        log.info('Saving %s special offers' % len(ready_offers))
        SpecialOffer.objects.all().delete()

        for offer in ready_offers:
            offer.save()

    else:
        log.info('Nothing to save')


def main():
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-m', '--check_maintenance', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    run()
