# -*- coding: utf-8 -*-
# Скрипт собирает файлы данных, полученных по WSDL, в архив
# и перемещает их в директорию с логами.

import travel.avia.admin.init_project  # noqa

import logging
import os.path
import shutil
import tarfile
from datetime import datetime, timedelta
from shutil import rmtree

from django.conf import settings

from travel.avia.admin.lib import tmpfiles
from travel.avia.admin.scripts import settings as script_settings
from travel.avia.admin.scripts.utils.import_file_storage import parse_date, get_stored_supplier_codes, get_all_temporary_dirs_by_dt, get_type_path
from travel.avia.admin.lib.logs import create_current_file_run_log
from travel.avia.library.python.common.utils import environment


log = logging.getLogger(__name__)


today = datetime.today()

LEFT_TABLO_DAYS = 1

# Удаляем файлы и архивы старше месяца
tablo_point_of_no_return_dt = today - timedelta(days=30)
schedule_point_of_no_return_dt = today - timedelta(days=30)

archive_path = script_settings.TABLO_IMPORT_ARCHIVE_PATH
if not os.path.exists(archive_path):
    os.makedirs(archive_path)

TAR_FORMAT = '%Y%m%d.tar.gz'


@tmpfiles.clean_temp
def compress_old_tablo_data():
    today = environment.today()
    to_compress_date = today - timedelta(LEFT_TABLO_DAYS)
    type_path = get_type_path('tablo')

    dirs = []

    for supplier_code in get_stored_supplier_codes('tablo'):
        for dt, path in get_all_temporary_dirs_by_dt('tablo', supplier_code).items():
            dirs.append((dt, path))

    to_compress_by_date = {}
    for dt, path in dirs:
        if dt.date() <= to_compress_date:
            to_compress_by_date.setdefault(dt.date(), []).append(path)

    for date_, files in to_compress_by_date.items():
        if not files:
            break

        tarfilepath = os.path.join(archive_path, date_.strftime(TAR_FORMAT))

        old_archive = None
        if os.path.exists(tarfilepath):
            old_archive = tmpfiles.get_tmp_filepath('archive.tar.gz')
            shutil.copy(tarfilepath, old_archive)

        with tarfile.open(tarfilepath, 'w:gz') as new_tar:
            if old_archive:
                with tarfile.open(old_archive) as old_tar:
                    for tarinfo in old_tar.getmembers():
                        if tarinfo.isfile():
                            fobj = old_tar.extractfile(tarinfo)
                            new_tar.addfile(tarinfo, fobj)
                        else:
                            new_tar.addfile(tarinfo)

            for path in files:
                new_tar.add(path, os.path.relpath(path, type_path))
                rmtree(path, ignore_errors=False)

            new_tar.close()


def clean_old():
    def get_date_from_tarfile(filepath):
        try:
            return datetime.strptime(os.path.basename(filepath), TAR_FORMAT)
        except ValueError:
            return datetime.fromtimestamp(os.stat(filepath).st_mtime)

    for tarfile_name in os.listdir(archive_path):
        tarfilepath = os.path.join(archive_path, tarfile_name)
        dt = get_date_from_tarfile(tarfilepath)
        if dt is not None and dt <= tablo_point_of_no_return_dt:
            os.remove(tarfilepath)

    temporary_dir = os.path.join(settings.DATA_PATH, 'schedule', 'temporary')
    if os.path.exists(temporary_dir):
        for supplier_dirname in os.listdir(temporary_dir):
            supplier_dir = os.path.join(temporary_dir, supplier_dirname)
            if not os.path.isdir(supplier_dir):
                continue

            for dt_dirname in os.listdir(supplier_dir):
                dt = parse_date(dt_dirname)
                dirpath = os.path.join(supplier_dir, dt_dirname)
                if dt is not None and dt <= schedule_point_of_no_return_dt:
                    rmtree(dirpath, ignore_errors=True)


def _main():
    compress_old_tablo_data()
    clean_old()


def main():
    create_current_file_run_log()

    _main()
