# -*- coding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
from datetime import timedelta

from django.db import transaction, connection

from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.models.geo import Station
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log


log = logging.getLogger(__name__)


cursor = connection.cursor()


def get_type_choices(station, enabled_type_choice_station):
    """Режимы, отображения, доступные для станции"""

    st_id = station.id

    suburban_type_station_key = ('suburban', st_id)
    aeroexpress_type_station_key = ('aeroexpress', st_id)
    train_type_station_key = ('train', st_id)

    if station.hidden or not station.has_traffic() or \
       (station.not_in_search() and not station.not_in_search_exact_mta()):
        return []

    if station.t_type.code == 'plane':
        types = ['tablo']

        if suburban_type_station_key in enabled_type_choice_station:
            types.append('suburban')

        if aeroexpress_type_station_key in enabled_type_choice_station:
            types.append('aeroexpress')

        return types

    elif station.t_type.code == 'train':
        types = []

        if station.in_tablo:
            types.append('tablo')

        if train_type_station_key in enabled_type_choice_station:
            types.append('train')

        if suburban_type_station_key in enabled_type_choice_station:
            types.append('suburban')

        if aeroexpress_type_station_key in enabled_type_choice_station:
            types.append('aeroexpress')

        return types

    elif station.t_type.code == 'bus':
        types = []

        if station.tablo_state in ('statuses', 'real'):
            types.append('tablo')

        types.append('schedule')

        return types

    else:
        return ['schedule']


@transaction.atomic
def fill_station_view_types():
    """Заполнение доступных типов отображения станции"""
    today = environment.today()

    # Отфильтровываем рейсы, не попадающие в промежуток (-3, +60) дней
    min_date = today - timedelta(days=3)
    max_date = today + timedelta(days=60)

    min_date_position = (min_date.month - 1) * 31 + min_date.day
    max_date_position = (max_date.month - 1) * 31 + max_date.day

    if max_date_position < min_date_position:
        max_date_position += 372

    enabled_type_choice_station_without_plane = set()

    log.info('Counting schedule subtypes')

    cursor.execute("""
SELECT DISTINCT
  st.id, t.express_type, tt.code
FROM
  www_station st INNER JOIN
  www_rtstation rts ON rts.station_id = st.id INNER JOIN
  www_rthread t ON t.id = rts.thread_id INNER JOIN
  www_transporttype tt ON t.t_type_id = tt.id,
  (SELECT
    %s AS first_day_pos,
    %s AS last_day_pos
  ) AS m
WHERE
  tt.code != 'plane' AND # Исключаем самолеты, т.к. не используем это в дальнейшем
  LOCATE('1', # Ходит ли нитка в какой либо день через станцию
    SUBSTRING(
    REPEAT(t.tz_year_days, 2)
    FROM m.first_day_pos
    FOR (m.last_day_pos - m.first_day_pos)
  )) > 0 AND
  # Исключаем станции, на которых нет остановки аэроэкспрессов
  (t.express_type != 'aeroexpress' OR rts.tz_arrival IS NULL OR rts.tz_departure IS NULL OR rts.tz_arrival < rts.tz_departure)
    """, [min_date_position, max_date_position])

    rows = cursor.fetchall()

    for row in rows:
        st_id = row[0]
        express_type = row[1]
        t_type_code = row[2]

        type_choice = express_type if express_type == 'aeroexpress' else t_type_code

        type_station_key = (type_choice, st_id)

        enabled_type_choice_station_without_plane.add(type_station_key)

    type_choices_ids = {}

    log.info('Counting type choices')

    for station in Station.objects.filter(hidden=False).select_related('t_type'):
        type_choices = ','.join(code for code in get_type_choices(station, enabled_type_choice_station_without_plane))

        type_choices_ids.setdefault(type_choices, []).append(station.id)

    log.info('Loading type choices')

    for type_choices, ids in type_choices_ids.items():
        type_choices = type_choices.replace('aeroexpress', 'aeroex')
        Station.objects.filter(id__in=ids).update(type_choices=type_choices)

    log.info('Done')


def main():
    create_current_file_run_log()

    from time import time
    start_t = time()
    from optparse import OptionParser

    parser = OptionParser()
    parser.add_option("-v", "--verbose", dest="verbose", action="store_true")

    (options, args) = parser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    fill_station_view_types()

    log.info(u'Done %s', time() - start_t)
