# -*- coding: utf-8 -*-
""" Хранит различные вспомогательные константы и методы для работы с данными.
    Нужен, чтобы избежать копипастов в заполняющих скриптах.
"""

import os
import time
import warnings
import logging
import codecs
from subprocess import call
from datetime import timedelta, date

from django.conf import settings
from django.db import transaction

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.admin.lib.exceptions import SimpleUnicodeException
from travel.avia.admin.scripts.schedule.utils.route_loader import CompactThreadNumberBuilder


log = logging.getLogger(__name__)


def get_aviacompany(code, log=log):
    from schedule.utils.plane import AviaCompanyFinder
    warnings.warn('[2015-08-06] Use AviaCompanyFinder', RaspDeprecationWarning, stacklevel=2)

    company_finder = AviaCompanyFinder()
    return company_finder.get_company(code)


def get_aviaroute_number(flight_number, company_code, company=None, log=log, fetch_company=False):
    if company is None and fetch_company:
        company = get_aviacompany(company_code, log)

    flight_number = flight_number.strip().lstrip(u'0')
    company_code = company_code.strip()

    if company is not None:
        route_number = (company.iata or company.icao or company.sirena_id or company_code) + \
            u' ' + flight_number

        return route_number

    return company_code.strip() + u" " + flight_number


def comma_separated_optparse_callback(option, opt_str, value, parser, type=unicode):
    values = filter(None, value.split(u','))
    values = map(lambda x: type(x.strip()), values)
    setattr(parser.values, option.dest, values)


def set_going(year_days, day, month, going):
    """ Ставит в дни хождения (year_days) в нужный день (day)
        нужного месяца (month) 1 или 0 (going) и возвращает результат.
    """
    month = int(month)
    day = int(day)
    if (
        month not in range(1, 13)
        or day not in range(1, 32)
    ):
        err_msg = " Неверный месяц или день: %s, %s!" % (str(month), str(day))
        raise Exception(err_msg)
    index = (month - 1) * 31 + day - 1
    year_days = year_days[:index] + str(going) + year_days[index + 1:]
    return year_days


def airport_search(code=None, title=None, title_en=None, not_raise=False):
    u""" Ищем аэропорт по коду, а затем по названию( русскому потом английскому)
    """
    stations = []
    if code:
        stations = Station.objects.filter(sirena_id=code)
        if not stations:
            stations = Station.objects.filter(code_set__system__code='iata',
                                                code_set__code=code)
        if not stations:
            stations = Station.objects.filter(code_set__system__code='icao',
                                                code_set__code=code)
    if not stations and title:
        stations = Station.objects.filter(title=title, t_type_id=TransportType.get_plane_type().id)
    if not stations and title_en:
        stations = Station.objects.filter(title_en=title_en,
                                          t_type_id=TransportType.get_plane_type().id)

    if len(stations) == 1:
        return stations[0]
    elif not stations:
        if not_raise:
            return None
        raise Station.DoesNotExist(code, title, title_en)
    elif len(stations) > 1:
        if not_raise:
            return None
        raise Station.MultipleObjectsReturned(code, title, title_en)


@transaction.atomic
def full_create_thread(thread, log=log, make_import_uid=True):
    """
    Добавляет нитку со станциями,
    станции должны лежать списком в аттрибуте rtstations
    Возвращает добавленную нитку.
    """

    if make_import_uid and not thread.import_uid:
        thread.gen_import_uid()

    thread.changed = True

    if not thread.ordinal_number:
        CompactThreadNumberBuilder([thread.route]).build_for(thread)

    thread.save(force_insert=True)

    if len(thread.rtstations) < 2:
        log.warning(u"Унитки %s %d станций", thread.uid, len(thread.rtstations))
    if thread.rtstations:
        for rts in thread.rtstations:
            rts.thread = thread
            rts.save(force_insert=True)

    log.info(u"Добавили нитку %s%s", thread.uid,
             u" #" + thread.template_text if thread.template_text else u"")
    return thread


@transaction.atomic
def full_create_route(route, log=log, make_import_uid=True):
    """
    Добавляет маршрут с нитками,
    нитки должны лежать списком в аттрибуте threads.
    Возвращает добавленный маршрут.
    """

    route.script_protected = False
    if not route.comment:
        route.comment = ''
    route.save(force_insert=True)
    for thread in route.threads:
        thread.route = route
        full_create_thread(thread, log, make_import_uid=make_import_uid)
    log.info(u"Маршрут %s добавлен успешно", route.route_uid)
    return route


def esr_leading_zeros(code):
    try:
        if len(code) != 3 and len(code) != 6:
            return u"%06d" % int(code)
    except ValueError:
        pass

    return code


def spk_leading_zeros(code):
    if len(code) != 4:
        code = u"%04d" % int(code)
    return code


def delete_old_export_files(template, older_then=1, log=log):
    u"""Удаляет позавчерашний(или старее чем указаон в older_then) экспорт и раньше.
    template - это шаблон, в который подставляется дата экспорта."""
    border = date.today() - timedelta(older_then)
    export_dir = os.path.dirname(template)
    log.debug(u"export_dir = %s", export_dir)
    for filename in os.listdir(export_dir):
        try:
            filepath = os.path.join(export_dir, filename)
            day = date(*time.strptime(filepath, template)[:3])
            if day < border:
                log.info(u"Удаляем старый файл экспорта %s", filepath)
                os.remove(filepath)
        except ValueError:
            # Не тот файл пропускаем
            pass


class StateSaver(object):
    def __init__(self, path):
        self.path = path

        dirpath = os.path.dirname(path)
        if not os.path.exists(dirpath):
            os.makedirs(dirpath)

    def set_state(self, state):
        f = codecs.open(self.path, 'w', encoding="utf8")
        f.write(state)
        f.close()

    def get_state(self):
        if os.path.exists(self.path):
            f = codecs.open(self.path, 'r', encoding="utf8")
            data = f.read().strip()
            f.close()

            return data
        else:
            return None

    def clean_state(self):
        if os.path.exists(self.path):
            os.remove(self.path)


def run_action_list(action_list, state_saver, log, on_error=None, is_continue=False):
    runner = ActionGroupListRunner(action_list, state_saver, log, on_error, is_continue)
    runner.run()
    return runner.failed_non_critical_actions


class ActionListCriticalError(SimpleUnicodeException):
    pass


class ActionListRunError(SimpleUnicodeException):
    pass


class ActionGroupListRunner():
    def __init__(self, action_groups, state_saver, log, on_error_callback=None, is_continue=False):
        self.action_groups = action_groups

        self.log = log
        self.on_error_callback = on_error_callback
        self.is_continue = is_continue

        self.state_saver = state_saver
        self.failed_non_critical_actions = []
        self.bad_action = None
        self.found_continuation = False
        self.last_state = None

        if self.is_continue:
            self.last_state = self.state_saver.get_state()
            if not self.last_state:
                self.log.warning(u"Последний статус пустой, запускаем скрипт заново")
            else:
                self.log.info(u"Пробуем продолжит скрипт с %s", self.last_state)
        else:
            self.state_saver.clean_state()

    def run(self):
        old_dir = os.path.normpath(os.getcwdu())

        try:
            return self.try_run()
        finally:
            os.chdir(old_dir)

    def try_run(self):
        for order, action in enumerate(self.action_iterator()):

            self.current_state = get_actionlist_state(action, order)

            if self.last_state and self.current_state != self.last_state:
                self.log.info(u"Пропускаем %s", self.current_state)
                continue

            elif self.last_state and self.current_state == self.last_state:
                self.log.info(u"Продолжаем скрипт с %s", self.current_state)
                self.last_state = None
                self.found_continuation = True

            self.state_saver.set_state(self.current_state)

            try:
                self.process_action(action)
            except ActionListCriticalError:
                self.log.critical(u"Критическая ошибка при выполнении списка задач, прерываем запуск!!!")
                break

        self.process_results()

    def process_results(self):
        if self.is_continue and not self.found_continuation:
            error_msg = u"Не смогли продолжить скрипт с %s" % self.last_state
            self.log.error(error_msg)

            self.on_error(error_msg)
            return

        if self.bad_action:
            self.log.error(u"Выполнение оборвалось на %s", self.bad_action)

            self.on_error(self.bad_action)

        else:
            self.state_saver.clean_state()

    def process_action(self, action):
        if isinstance(action, basestring):
            self.log.info(self.current_state)
        else:
            self.log.info(u'Запускаем %s' % self.current_state)

            critical, real_action = action
            try:
                self.run_action(real_action)
                self.log.info(u"%s отработал успешно", self.current_state)
            except ActionListRunError:
                if critical:
                    self.bad_action = self.current_state
                    raise ActionListCriticalError(self.current_state)
                else:
                    self.failed_non_critical_actions.append(self.current_state)
                    self.log.error(u"%s failed", self.current_state)

    def run_action(self, real_action):
        if callable(real_action):
            try:
                real_action()
            except Exception:
                self.log.exception(u"Ошибка при выполнении функции %s", real_action.__name__)
                raise ActionListRunError(self.current_state)
        else:
            ret_code = call(real_action)
            if ret_code != 0:
                self.log.error(u"Скрипт упал!!! С кодом %s" % ret_code)
                raise ActionListRunError(self.current_state)

    def action_iterator(self):
        for action_group in self.action_groups:
            if isinstance(action_group, tuple):
                params, actions = action_group
                self.process_params(params)
                for action in actions:
                    yield action
            else:
                ValueError("Unexpected action type %s" % type(action_group))

    def on_error(self, error_msg):
        if self.on_error_callback:
            return self.on_error_callback(error_msg)

    def process_params(self, params):
        self.params = params
        if 'base_path' in params:
            self.log.info(u"Устанавливаем текущую директорию в %s", params['base_path'])
            os.chdir(params['base_path'])


def get_actionlist_state(action, order):
    status_line = u"%d " % order
    if isinstance(action, basestring):
        return status_line + u"notification " + action

    critical, script_or_function = action
    if critical:
        status_line += u"critical "

    if callable(script_or_function):
        return status_line + u"function %s" % script_or_function.__name__
    else:
        return status_line + u"script " + u" ".join(script_or_function)


def get_state_path(filename):
    return os.path.join(settings.LOG_PATH, 'run_states', filename)
