# -*- coding: utf-8 -*-

# 1 континент
# 2 регион
# 3 страна
# 4 федеральный округ
# 5 субъект федерации<!--  (края, автономные округа, области, республики, а также любые части иностранных государств (Крым, Гонконг и т.п.)) -->
# 6 город
# 7 село
# 8 район города
# 9 станция метро
# 10 район субъекта федерации
# 11 аэропорт
# 0 Прочее/универсальное
# -1 Скрытые

import travel.avia.admin.init_project  # noqa

import os.path
import gzip
import re
import logging
from optparse import OptionParser

import pytz
from django.utils.functional import cached_property
from django.conf import settings

from travel.avia.library.python.common.models.geo import Settlement, Region, Country, CityMajority
from travel.avia.library.python.common.models.translations import TranslatedTitle
from travel.avia.admin.lib.logs import create_current_file_run_log, print_log_to_stdout, get_collector_context
from travel.avia.library.python.common.lib.mail import mail_datachanges
from travel.avia.admin.www.utils.data import change_similar


log = logging.getLogger(__name__)


class GeobaseUpdater(object):
    def __init__(self):
        self.simple_city_majority = self.default_majority = CityMajority.objects.get(title=u'Обычный город')
        self.country_capital_majority = CityMajority.objects.get(title=u'Столица страны')
        self.region_capital_majority = CityMajority.objects.get(title=u'Столица области')
        self.millionaire_city_majority = CityMajority.objects.get(title=u'Город-миллионник')
        self.village_majority = CityMajority.objects.get(title=u'Посёлок или деревня')

    @cached_property
    def geobase(self):
        import geobase6 as geobase_lib
        return geobase_lib.Lookup('/var/cache/geobase/geodata6.bin')

    @cached_property
    def populations(self):
        with gzip.open(os.path.join(settings.DATA_PATH, 'geobase_populations.tsv.gz')) as populations_file:
            populations_file.next()
            lines = [line.split() for index, line in enumerate(populations_file) if line.strip()]

            return {int(id_): int(pop) for id_, pop in lines}

    def update(self):
        log.info('Start')
        for geo_country in self.geobase.regions_by_type(3):  # страны
            country = self.update_country(geo_country)

            for geo_region in self.get_children(geo_country, [5]):
                region = self.update_region(geo_region, country)

                for geo_settlement in self.get_children(geo_region, [6, 7]):
                    self.update_settlement(geo_settlement, geo_region, geo_country, region, country)

            for geo_settlement in self.get_children(geo_country, [6, 7]):
                self.update_settlement(geo_settlement, None, geo_country, None, country)  # Города без областей

        log.info('Finish')

    def get_children(self, geo_point, types):
        for child_id in self.geobase.children(geo_point.id):
            point = self.geobase.region_by_id(child_id)
            if point.type in types:
                yield point
            # 10 == Район субъекта федерации.
            elif geo_point.type == 5 and point.type == 10:
                for point in self.get_children(point, types):
                    yield point
            # 4 == Федеральный округ.
            elif geo_point.type == 3 and point.type == 4:
                for point in self.get_children(point, types):
                    yield point

    def update_country(self, geo_country):
        """Ищет страну по параметрам, если не находит - создает"""
        country_name = geo_country.name.decode('utf-8')
        log.debug(u'Country "%s"', country_name)

        geo_id = geo_country.id
        title = fix_title(country_name, geo_id)

        title_en = geo_country.ename.decode('utf8').strip() if geo_country.ename else None

        try:
            country = Country.objects.get(_geo_id=geo_id)
            if country.title != title:
                log.warning(u'Название страны "%s" (%s) не совпадает с данными в нашей базе (%s)',
                            title, geo_id, country.title)

        except Country.DoesNotExist:
            try:
                country = Country.objects.get(title=title)

            except Country.DoesNotExist:
                log.info(u'Создаем страну %s (%s)', title, geo_id)
                country = Country(title=title, title_en=title_en, _geo_id=geo_id, prefix_in=u'в')

        if not country._geo_id:
            log.info(u'Выставляем geo_id %s стране %s', geo_id, title)
            country._geo_id = geo_id

        country.save()
        return country

    def update_region(self, geo_region, country):
        """Ищет область по параметрам, если не находит - создает"""
        region_name = geo_region.name.decode('utf-8')
        log.debug(u'Region "%s"', region_name)
        geo_id = geo_region.id
        title = fix_title(region_name, geo_id)

        try:
            region = Region.objects.get(_geo_id=geo_id)

            if region.title != title:
                log.warning(u'Название региона "%s" (%s) не совпадает с данными в нашей базе (%s)',
                            title, geo_id, region.title)

        except Region.DoesNotExist:
            if not country:
                log.error(u'Не обновляем регион "%s" (%s), т.к. не нашли по geo_id и у региона не указана страна',
                          title, geo_id)
                return

            regions = Region.objects.filter(title=title, country=country)
            if len(regions) > 1:
                log.warning(u'Несколько областей с названием "%s" для кода "%s", пропускаем', title, geo_id)
                return

            elif len(regions) == 1:
                region = regions[0]

            else:
                log.info(u'Создаем область "%s" (%s)', title, geo_id)
                title_en = geo_region.ename.decode('utf8').strip() if geo_region.ename else None
                translated_title = TranslatedTitle(
                    ru_nominative=title,
                )
                translated_title.save()
                region = Region(
                    new_L_title=translated_title,
                    country=country,
                    title=title,
                    title_en=title_en,
                    _geo_id=geo_id
                )

        if not region.title:
            region.title = title

        if not region._geo_id:
            log.info(u'Выставляем geo_id "%s" области "%s"', geo_id, title)
            region._geo_id = geo_id

        if not region.time_zone:
            time_zone = get_timezone(geo_region)
            if time_zone in pytz.all_timezones_set:
                region.time_zone = time_zone
                log.info(u'Выставляем time_zone "%s" области "%s" (%s)', region.time_zone, title, geo_id)

        region.save()
        return region

    def get_recomended_majority(self, geo_settlement, geo_region, geo_country):
        if geo_settlement.id == geo_country.chief_region:
            return self.country_capital_majority

        if geo_region and geo_settlement.id == geo_region.chief_region:
            return self.region_capital_majority

        # 6 город
        # 7 село
        if geo_settlement.type == 6:
            if self.populations.get(geo_settlement.id, 0) >= 1000000:
                return self.millionaire_city_majority
            else:
                return self.simple_city_majority
        elif geo_settlement.type == 7:
            return self.village_majority

        return self.default_majority

    def update_settlement(self, geo_settlement, geo_region, geo_country, region, country):
        """Ищет город по параметрам, если не находит - создает"""
        settlement_name = geo_settlement.name.decode('utf-8')
        geo_id = geo_settlement.id
        log.debug(u'Settlement "%s" %s', settlement_name, geo_id)
        title = cut_title(fix_title(settlement_name, geo_id))
        settlement = None

        title_en = geo_settlement.ename.decode('utf8').strip() if geo_settlement.ename else None

        recommended_majority = self.get_recomended_majority(geo_settlement, geo_region, geo_country)

        try:
            settlement = Settlement.objects.get(_geo_id=geo_id)

            if settlement.majority_id != recommended_majority.id:
                log.warning(u'Не совпадает рекомендуемая важность и важность в базе %s; %s; %s; %s; рекомендуемая %s',
                            settlement._geo_id, settlement.id, settlement.title, settlement.majority.title,
                            recommended_majority.title)

            if settlement.title != title:
                log.warning(u'Название города "%s" (%s) не совпадает с данными в нашей базе ("%s") [%s]',
                            title, geo_id, settlement.title, settlement.L_full_geography())

            if settlement.region and region and settlement.region != region:
                log.warning(u'Регион города %s (%s) (%s) не совпадает с данными в нашей базе (%s) [%s]',
                            title, region.title, geo_id, settlement.region.title, settlement.L_full_geography())

        except Settlement.DoesNotExist:
            # Пробуем найти по имени в области
            try:
                settlement = Settlement.objects.get(title=title, region=region, region__isnull=False)

            except Settlement.MultipleObjectsReturned:
                log.error(u'В регионе "%s" найдено больше одного города с названием "%s"',
                          region and region.title or '', title)
                return

            except Settlement.DoesNotExist:
                # Пробуем найти в стране остальные по имени
                settlements = Settlement.objects.filter(title=title, country=country, country__isnull=False)

                # Если город один и у него не указан субъект федерации, то считаем его нужным нам
                if len(settlements) == 1 and settlements[0].region is None:
                    settlement = settlements[0]

                if settlements:
                    log.warning(u'Города с названием %s (%s) есть кроме %s в областях: %s' % (
                        title, geo_id,
                        region and region.title or u'-',
                        u', '.join(s.region and s.region.title or u'(страна)' for s in settlements)))

        if not settlement:
            settlement = Settlement(country=country,
                                    region=region,
                                    title=title,
                                    title_en=title_en,
                                    _geo_id=geo_id,
                                    majority=recommended_majority)
            log.info(u'Создаем город %s (geo_id %s) [%s]', title, geo_id, settlement.L_full_geography())

        if not settlement.title:
            settlement.title = title

        if not settlement._geo_id:
            log.info(u'Выставляем geo_id %s городу %s [%s]', geo_id, title, settlement.L_full_geography())
            settlement._geo_id = geo_id

        if region and not settlement.region:
            log.info(u'Выставляем область "%s" городу %s [%s] (%s)',
                     region, title, settlement.L_full_geography(), geo_id)
            settlement.region = region

        if not settlement.time_zone:
            time_zone = get_timezone(geo_settlement)

            if time_zone in pytz.all_timezones_set:
                log.info(u'Выставляем time_zone "%s" городу %s [%s] (%s)',
                         time_zone, title, settlement.L_full_geography(), geo_id)
                settlement.time_zone = time_zone

        settlement.save()


def fix_title(title, geo_id):
    similar_title = change_similar(title, 'rus')

    if title != similar_title:
        title_with_highlights = u''.join(
            u'({})'.format(old_ch) if old_ch != similar_ch else old_ch
            for old_ch, similar_ch in zip(title, similar_title)
        )
        log.warning(u'Латинские буквы в названии региона в геобазе %s: %s', geo_id, title_with_highlights)

        title = similar_title

    if title != title.strip():
        log.warning(u'Пробелы в названии региона в геобазе %s: "%s"', geo_id, title)

    return title.strip()


def cut_title(title):
    """В геобазе в название города иногда включается область в скобках, нужно обрезать"""
    result = re.match(r'^(.*)\s+\(.*\)$', title)
    if result:
        title = result.group(1)
    return title


def get_timezone(point):
    if point.timezone:
        return point.timezone


def main():
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true', help=u'выводить лог на экран')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    with get_collector_context(log) as collector:
        updater = GeobaseUpdater()
        updater.update()

        mail_datachanges(u'Изменения при обновлении геобазы', collector.get_collected())
