# -*- coding: utf-8 -*-

import os.path
import shutil
from datetime import datetime

from django.conf import settings
from django.utils.datastructures import SortedDict
from travel.avia.library.python.common.utils import environment


PATH_DATE_FORMAT = '%Y-%m-%d'
PATH_DATETIME_FORMAT = '%Y-%m-%d_%H%M%S'


def parse_date(dirname):
    try:
        return datetime.strptime(dirname, PATH_DATETIME_FORMAT)
    except ValueError:
        pass

    try:
        return datetime.strptime(dirname, PATH_DATE_FORMAT)
    except ValueError:
        pass


def get_schedule_temporary_path(supplier_or_package):
    subpath = get_subpath_by_supplier_or_package(supplier_or_package)

    dir_path = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', subpath)

    if not os.path.exists(dir_path):
        os.makedirs(dir_path)

    return dir_path


def get_schedule_temporary_date_filepath(filename, supplier_or_package, today=None):
    today = today or environment.today()

    dir_path = os.path.join(
        get_schedule_temporary_path(supplier_or_package),
        today.strftime(PATH_DATE_FORMAT)
    )

    if not os.path.exists(dir_path):
        os.makedirs(dir_path)

    return os.path.join(dir_path, filename)


def remove_schedule_temporary_today_dir(supplier_or_package, today=None):
    subpath = get_subpath_by_supplier_or_package(supplier_or_package)

    today = today or environment.today()

    dir_path = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', subpath,
                            today.strftime(PATH_DATE_FORMAT))

    if not os.path.exists(dir_path):
        return

    error_files = []

    def on_error(_f, path, exc_info):
        error_files.append(path)

    shutil.rmtree(dir_path, onerror=on_error)

    return error_files


def remove_cysix_xml(supplier_or_package, today=None):
    subpath = get_subpath_by_supplier_or_package(supplier_or_package)
    today = today or environment.today()

    filepath = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', subpath,
                            today.strftime(PATH_DATE_FORMAT), 'cysix.xml')

    os.remove(filepath)


def get_subpath_by_supplier_or_package(supplier_or_package):
    if hasattr(supplier_or_package, 'code'):
        return supplier_or_package.code

    else:
        return '%s/%s' % (supplier_or_package.supplier.code, supplier_or_package.id)


def get_schedule_filepath(filename, supplier):
    dir_path = os.path.join(settings.DATA_PATH, 'schedule', supplier.code)
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_schedule_temporary_datetime_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', supplier.code,
                            now.strftime(PATH_DATETIME_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_tablo_temporary_datetime_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'tablo', 'temporary', supplier.code,
                            now.strftime(PATH_DATETIME_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_tablo_temporary_date_filepath(filename, supplier, now=None):
    now = now or environment.now()

    dir_path = os.path.join(settings.DATA_PATH, 'tablo', 'temporary', supplier.code,
                            now.strftime(PATH_DATE_FORMAT))
    dir_path = str(dir_path)
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)
    return os.path.join(dir_path, filename)


def get_last_temporary_date_file(filename, supplier):
    for date_, filepath in get_all_temporary_date_files(filename, supplier).items():
        return filepath


def get_all_temporary_date_files(search_filename, supplier):
    u'''
    Возвращает итератор на дату, файл в начале файл последней версии
    @param search_filename str:
    @param supplier www.models.Supplier():
    '''
    supplier_dir = os.path.join(settings.DATA_PATH, 'schedule', 'temporary', supplier.code)
    if os.path.exists(supplier_dir):
        dates = os.listdir(supplier_dir)
    else:
        dates = []

    by_dates = SortedDict()
    for date_dir in dates:
        if os.path.isdir(os.path.join(supplier_dir, date_dir)):
            try:
                date_ = datetime.strptime(date_dir, PATH_DATE_FORMAT).date()
                by_dates[date_] = None
            except ValueError:
                pass

    for date_ in list(by_dates.keys()):
        file_dir = os.path.join(supplier_dir, date_.strftime(PATH_DATE_FORMAT))
        for filename in os.listdir(file_dir):
            if filename == search_filename:
                by_dates[date_] = os.path.join(file_dir, filename)

    by_dates = SortedDict(((k, v) for k, v in by_dates.items() if v is not None))
    by_dates.keyOrder.sort(reverse=True)
    return by_dates


def get_stored_supplier_codes(type_):
    type_path = get_type_path(type_)

    if not os.path.isdir(type_path):
        return []

    return [
        d
        for d in os.listdir(type_path)
        if os.path.isdir(os.path.join(type_path, d))
    ]


def get_type_path(type_):
    return os.path.join(settings.DATA_PATH, type_, 'temporary')


def get_all_temporary_dirs_by_dt(type_, supplier):
    supplier_code = supplier if isinstance(supplier, basestring) else supplier.code

    supplier_dir = os.path.join(get_type_path(type_), supplier_code)

    dates = os.listdir(supplier_dir)
    result = SortedDict()

    def get_dt(dir_name):
        try:
            return datetime.strptime(dir_name, PATH_DATE_FORMAT)
        except ValueError:
            pass

        try:
            return datetime.strptime(dir_name, PATH_DATETIME_FORMAT)
        except ValueError:
            pass

    for dt_dir_name in dates:
        dt_dir_path = os.path.join(supplier_dir, dt_dir_name)
        if os.path.isdir(dt_dir_path):
            dt = get_dt(dt_dir_name)

            if dt is not None:
                result[dt] = dt_dir_path

    result.keyOrder.sort(reverse=True)

    return result


def mark_error(filepath, log):
    dirpath, filename = os.path.split(filepath)
    new_filename = 'error_' + filename
    new_path = os.path.join(dirpath, new_filename)
    os.rename(filepath, new_path)
    log.error(u"Пометили файл %s как ошибочный. %s", filepath, new_path)
